import React, {MutableRefObject, useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {ITrainsPreviousSearch} from 'types/trains/previousSearches/IPreviousSearch';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import fillSearchFormByPreviousSearchThunkAction from 'reducers/trains/previousSearches/thunk/fillSearchFormByPreviousSearch';

import {reachGoal} from 'utilities/metrika';
import {trainsURLs} from 'projects/trains/lib/urls';
import {getTrainsHumanizeDatesLabel} from 'projects/trains/lib/date/getTrainsHumanizeDatesLabel';

import * as i18nBlock from 'i18n/common';

import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import SearchPreset from 'components/SearchPreset/SearchPreset';

interface IPreviousSearchProps {
    item: ITrainsPreviousSearch;
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
    index: number;
}

const PreviousSearch: React.FC<IPreviousSearchProps> = props => {
    const {item, searchFormFieldsRef, index} = props;
    const {startDate, endDate} = item;
    const history = useHistory();
    const dispatch = useDispatch();

    const handleClick = useCallback(() => {
        dispatch(fillSearchFormByPreviousSearchThunkAction(item));

        if (item.startDate) {
            history?.push(
                trainsURLs.getTrainsSearchUrl({
                    context: {
                        from: item.from.slug,
                        to: item.to.slug,
                        when: item.startDate,
                        returnWhen: item.endDate,
                    },
                }),
            );
        } else {
            searchFormFieldsRef.current?.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        }

        reachGoal(ETrainsGoal.PREVIOUS_SEARCHES_PRESET_CLICK);
    }, [dispatch, item, searchFormFieldsRef, history]);

    const searchDatesLabel = useMemo<string | undefined>(() => {
        return getTrainsHumanizeDatesLabel({startDate, endDate});
    }, [startDate, endDate]);

    return (
        <SearchPreset
            onClick={handleClick}
            title={i18nBlock.direction({
                from: item.from.title,
                to: item.to.title,
            })}
            theme="blue"
            secondRow={searchDatesLabel}
            index={index}
        />
    );
};

export default React.memo(PreviousSearch);
