import React, {ReactNode, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    IGetContextPageArticle,
    IGetContextPageTag,
} from 'server/api/TemplatorApi/types/IGetContextPageResponse';

import {deviceMods} from 'utilities/stylesUtils';
import {mapTags} from 'projects/journal/utilities/mapTags';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {usePaginationWithHistory} from 'components/Pagination/utilities/usePaginationWithHistory';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Pagination from 'components/Pagination/Pagination';
import ArticleCard from 'projects/journal/components/ArticleCard/ArticleCard';

import cx from './Articles.scss';

interface IArticlesAdditionalContent {
    insertIndex: number;
    content: React.ReactElement;
}

interface IArticlesProps extends IWithClassName {
    articles: IGetContextPageArticle[];
    tags?: IGetContextPageTag[];
    articlesPerPage: number;
    additionalContent?: IArticlesAdditionalContent;
}

const PAGE_PARAM_NAME = 'page';

const Articles: React.FC<IArticlesProps> = props => {
    const {articles, articlesPerPage, additionalContent, tags, className} =
        props;
    const deviceType = useDeviceType();

    const maxPageNumber = Math.ceil(articles.length / articlesPerPage);
    const [page, handlePageChange] = usePaginationWithHistory(
        PAGE_PARAM_NAME,
        1,
        maxPageNumber,
    );

    const startArticleNumber = (page - 1) * articlesPerPage;
    const articlesToShow = useMemo(
        () =>
            articles.slice(
                startArticleNumber,
                startArticleNumber + articlesPerPage,
            ),
        [articles, articlesPerPage, startArticleNumber],
    );
    const separatorIndex = additionalContent
        ? additionalContent.insertIndex +
          (deviceType.isMobile && page === 1 ? 1 : 0)
        : 0;
    const articlesBeforeAdditionalContent = useMemo(
        () => articlesToShow.slice(0, separatorIndex),
        [articlesToShow, separatorIndex],
    );
    const articlesAfterAdditionalContent = useMemo(
        () => articlesToShow.slice(separatorIndex),
        [articlesToShow, separatorIndex],
    );
    const tagsMap = useMemo(
        () =>
            tags
                ? articles.reduce<Record<string, IGetContextPageTag[]>>(
                      (acc, item) => ({
                          ...acc,
                          [item.semanticId]: mapTags(item.tags, tags),
                      }),
                      {},
                  )
                : {},
        [articles, tags],
    );

    const renderArticles = (
        articlesToRender: IGetContextPageArticle[],
    ): ReactNode =>
        articlesToRender.map(article => (
            <ArticleCard
                className={cx('article')}
                key={article.semanticId}
                semanticId={article.semanticId}
                imageSrc={article.articleImg}
                title={article.pageTitle}
                author={article.author}
                publishDate={article.createdDate}
                tags={tagsMap[article.semanticId]}
            />
        ));

    const renderRows = (
        articlesToRender: IGetContextPageArticle[],
        groupCount: number = 2,
    ): ReactNode =>
        articlesToRender
            .reduce<IGetContextPageArticle[][]>(
                (acc, item) => {
                    if (acc[acc.length - 1].length < groupCount) {
                        acc[acc.length - 1].push(item);
                    } else {
                        acc.push([item]);
                    }

                    return acc;
                },
                [[]],
            )
            .map((row, index) => (
                <Flex
                    className={cx('row')}
                    key={index}
                    between="10"
                    flexDirection="row"
                    inline
                    nowrap
                >
                    {renderArticles(row)}
                </Flex>
            ));

    if (!articles.length) {
        return null;
    }

    return (
        <Flex
            className={cx('root', className, deviceMods('root', deviceType))}
            between="10"
            flexDirection="column"
            justifyContent="flex-start"
        >
            {deviceType.isMobile ? (
                <Box between="8">
                    {renderArticles(articlesBeforeAdditionalContent)}
                    {additionalContent?.content}
                    {renderArticles(articlesAfterAdditionalContent)}
                </Box>
            ) : (
                <Box>
                    <Box between="10">
                        {renderRows(articlesBeforeAdditionalContent)}
                    </Box>
                    {additionalContent?.content}
                    <Box between="10">
                        {renderRows(articlesAfterAdditionalContent)}
                    </Box>
                </Box>
            )}

            <Pagination
                page={page}
                itemsCount={articles.length}
                itemsPerPage={articlesPerPage}
                onPageChange={handlePageChange}
            />
        </Flex>
    );
};

export default Articles;
