import React, {useCallback} from 'react';

import {INavigationItem} from 'types/common/INavigationItem';
import {EJournalGoals} from 'utilities/metrika/types/goals/journal';

import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Box from 'components/Box/Box';
import Hero from 'projects/journal/components/Hero/Hero';
import Navigation from 'components/Navigation/Navigation';
import TwoColumnLayout from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import HeroContent from 'projects/journal/components/HeroContent/HeroContent';

import CommonLayout from '../CommonLayout/CommonLayout';

import cx from './ContentLayout.scss';

interface IContentLayoutProps {
    /** Источник картинки */
    imageSrc: string;
    /** Альтернативная подпись к картинке */
    imageAlt: string;
    /** Заголовок статьи */
    title: string;
    /** */
    navigation?: INavigationItem[];
}

const ContentLayout: React.FC<IContentLayoutProps> = props => {
    const {imageSrc, imageAlt, title, navigation, children} = props;
    const deviceType = useDeviceType();

    const onLinkClick = useCallback(
        (
            e: React.MouseEvent<HTMLElement, MouseEvent> & {
                target: HTMLAnchorElement;
            },
        ) => {
            const anchor = e.target.getAttribute('href');

            if (anchor) {
                reachGoal(EJournalGoals.JOURNAL_ARTICLE_ANCHOR_CLICK, {
                    journal: {articleAnchor: anchor},
                });
            }
        },
        [],
    );

    return (
        <CommonLayout>
            <Hero
                className={cx('hero', deviceMods('hero', deviceType))}
                src={imageSrc}
                alt={imageAlt}
                height={deviceType.isMobile ? 386 : 400}
                contentClassName={cx(
                    'heroContent',
                    deviceMods('heroContent', deviceType),
                )}
            >
                <HeroContent title={title} />
            </Hero>

            {deviceType.isMobile && navigation && (
                <Navigation
                    className={cx('navigation', 'navigation_mobile')}
                    items={navigation}
                    onLinkClick={onLinkClick}
                    expandable
                />
            )}

            <Box
                className={cx('content', deviceMods('content', deviceType))}
                y={deviceType.isDesktop ? 10 : 6}
            >
                {deviceType.isMobile ? (
                    children
                ) : (
                    <TwoColumnLayout
                        className={cx('columns')}
                        rightColumnOffset={16}
                        rightColumnWidth={64}
                        deviceType={deviceType}
                    >
                        <TwoColumnLayout.LeftColumn
                            className={cx('leftColumn')}
                        >
                            <Box className={cx('content')}>{children}</Box>
                        </TwoColumnLayout.LeftColumn>
                        <TwoColumnLayout.RightColumn>
                            {navigation && (
                                <Navigation
                                    className={cx(
                                        'navigation',
                                        'navigation_desktop',
                                    )}
                                    items={navigation}
                                    onLinkClick={onLinkClick}
                                />
                            )}
                        </TwoColumnLayout.RightColumn>
                    </TwoColumnLayout>
                )}
            </Box>
        </CommonLayout>
    );
};

export default ContentLayout;
