import React from 'react';
import {Link} from 'react-router-dom';

import {IWithClassName} from 'types/withClassName';
import {IJournalCardProps} from 'projects/journal/types/IJournalCardProps';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';
import {getArticleUrl} from 'projects/journal/utilities/getArticleUrl';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import TravelImage from 'components/TravelImage/TravelImage';
import CardAuthor from 'projects/journal/components/CardAuthor/CardAuthor';
import TagsOverlay from 'projects/journal/components/TagsOverlay/TagsOverlay';

import cx from './HeaderCard.scss';

export interface IHeaderCardProps extends IJournalCardProps, IWithClassName {}

const HeaderCard: React.FC<IHeaderCardProps> = props => {
    const deviceType = useDeviceType();
    const query = useQueryByLocation();

    return (
        <Box
            className={cx(
                'root',
                deviceMods('root', deviceType),
                props.className,
            )}
            between="4"
        >
            <div className={cx('head')}>
                <TravelImage
                    className={cx('image')}
                    src={props.imageSrc}
                    imageAlt={props.imageAlt}
                    hasFitCover
                />

                {props.tags && (
                    <TagsOverlay className={cx('tags')} tags={props.tags} />
                )}
            </div>

            <Flex
                flexDirection="row"
                justifyContent="space-between"
                alignItems="flex-start"
                inline
                between="8"
                nowrap
            >
                <Heading
                    level="2"
                    className={cx('title', deviceMods('title', deviceType))}
                >
                    {props.title}
                </Heading>

                <CardAuthor
                    className={cx('author')}
                    author={props.author}
                    date={props.publishDate}
                />
            </Flex>

            <Link
                className={cx('link')}
                to={getArticleUrl(props.semanticId, query)}
            />
        </Box>
    );
};

export default HeaderCard;
