import React, {useCallback, useState} from 'react';
import {useLocation} from 'react-router-dom';

import {EJournalGoals} from 'utilities/metrika/types/goals/journal';
import {EJournalArticleShareClickTarget} from 'utilities/metrika/types/params/journal';

import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Box from 'components/Box/Box';
import Share from 'components/Share/Share';
import Heading from 'components/Heading/Heading';
import ShareFieldIcon from 'icons/16/ShareFilled';
import Breadcrumbs from 'projects/journal/components/Breadcrumbs/Breadcrumbs';

import cx from './HeroContent.scss';

interface IHeroContentProps {
    title: string;
}

const HeroContent: React.FC<IHeroContentProps> = ({title}) => {
    const {pathname, hash} = useLocation();
    const deviceType = useDeviceType();
    const [url, setUrl] = useState<string>('');

    const handleShareClick = useCallback(() => {
        setUrl(
            new URL(`${pathname}${hash}`, window.location.origin).toString(),
        );
    }, [pathname, hash, setUrl]);

    const onInputFocus = useCallback(() => {
        reachGoal(EJournalGoals.JOURNAL_ARTICLE_SHARE_CLICK, {
            journal: {
                articleShareClickTarget: EJournalArticleShareClickTarget.LINK,
            },
        });
    }, []);

    const onShareLinkClick = useCallback((text?: string) => {
        if (text) {
            reachGoal(EJournalGoals.JOURNAL_ARTICLE_SHARE_CLICK, {
                journal: {articleShareClickTarget: text},
            });
        }
    }, []);

    return (
        <Box className={cx('root')} between="1">
            <Breadcrumbs />

            <Heading
                className={cx('title', deviceMods('title', deviceType))}
                level={1}
            >
                {title}
            </Heading>

            <Share
                className={cx('share')}
                button={<ShareFieldIcon className={cx('icon')} />}
                buttonWrapperClassName={cx('button')}
                link={url}
                onClick={handleShareClick}
                onInputFocus={onInputFocus}
                onShareLinkClick={onShareLinkClick}
                page="journal"
            />
        </Box>
    );
};

export default HeroContent;
