import React, {useCallback, useEffect, useMemo} from 'react';
import {useSelector} from 'react-redux';
import {useInView} from 'react-intersection-observer';
import {generatePath, useParams} from 'react-router-dom';
import moment from 'moment';

import {URLs} from 'constants/urls';
import {MINUTE} from 'utilities/dateUtils/constants';
import {PRODUCTION_PROJECT_URL} from 'constants/common';
import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';

import {
    isCTALinkBlock,
    isImageBlock,
    isMapBlock,
    isMarkupBlock,
} from 'projects/journal/types/article/blocks';
import {EJournalGoals} from 'utilities/metrika/types/goals/journal';
import {
    EAuthorType,
    IGetContextPageTag,
} from 'server/api/TemplatorApi/types/IGetContextPageResponse';

import journalTagsListSelector from 'selectors/journal/journalTagsListSelector';
import journalArticlePageSelector from 'selectors/journal/journalArticlePageSelector';

import {internalUrl} from 'utilities/url';
import {reachGoal} from 'utilities/metrika';
import {formatDate} from 'utilities/dateUtils';
import {loadable} from 'utilities/pageLoadable';
import {deviceMods} from 'utilities/stylesUtils';
import {HUMAN_DATE_RU, HUMAN_WITH_YEAR} from 'utilities/dateUtils/formats';
import {mapTags} from 'projects/journal/utilities/mapTags';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useTimeOnTab} from 'utilities/hooks/useTimeOnTab';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';
import {buildNavigation} from 'projects/journal/utilities/buildNavigation';
import {useFetchArticle} from 'projects/journal/pages/Article/utilities/useFetchArticle';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Tags from 'projects/journal/components/Tags/Tags';
import MapBlock from 'projects/journal/pages/Article/components/MapBlock/MapBlock';
import ImageBlock from 'projects/journal/pages/Article/components/ImageBlock/ImageBlock';
import MarkupBlock from 'projects/journal/pages/Article/components/MarkupBlock/MarkupBlock';
import ArticleMeta from 'projects/journal/pages/Article/components/ArticleMeta/ArticleMeta';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import ContentLayout from 'projects/journal/components/ContentLayout/ContentLayout';
import JournalPageMeta from 'projects/journal/components/JournalPageMeta/JournalPageMeta';
import CTALink from 'projects/journal/pages/Article/components/CTALink/CTALink';
import Author from './components/Author/Author';
import ZenAuthorFooter from './components/ZenAuthorFooter/ZenAuthorFooter';

import cx from './Article.scss';

const NotFoundApp = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

const TWO_MINUTES = MINUTE * 2;

const Article: React.FC = () => {
    const deviceType = useDeviceType();
    const query = useQueryByLocation();
    const {articleSlug} = useParams<{articleSlug: string}>();
    const {data, isFetched} = useSelector(journalArticlePageSelector);
    const {data: tagsList} = useSelector(journalTagsListSelector);
    const tags = useMemo<IGetContextPageTag[]>(() => {
        if (!isFetched || !data) {
            return [];
        }

        return mapTags(data.tags, tagsList);
    }, [data, isFetched, tagsList]);

    const [ref, wasInView] = useInView({
        threshold: 0.2,
        triggerOnce: true,
    });

    useFetchArticle();

    const useTimeOnTabCallback = useCallback(() => {
        reachGoal(EJournalGoals.JOURNAL_ARTICLE_USER_ON_PAGE_TWO_MINUTES);
    }, []);

    useTimeOnTab({timeout: TWO_MINUTES, callback: useTimeOnTabCallback});

    useEffect(() => {
        if (wasInView) {
            reachGoal(EJournalGoals.JOURNAL_ARTICLE_LAST_BLOCK_WAS_SHOWN);
        }
    }, [wasInView]);

    const onLinkClick = useCallback(
        (
            e: React.MouseEvent<HTMLDivElement, MouseEvent> & {
                target: HTMLAnchorElement;
            },
        ) => {
            if (e.target.tagName.toLowerCase() === 'a') {
                if (
                    e.target.href.startsWith(PRODUCTION_PROJECT_URL) ||
                    e.target.href.startsWith('/')
                ) {
                    reachGoal(
                        EJournalGoals.JOURNAL_ARTICLE_INTERNAL_LINK_CLICK,
                    );
                } else {
                    reachGoal(EJournalGoals.JOURNAL_ARTICLE_LINK_CLICK, {
                        journal: {articleLinkPath: e.target.href},
                    });
                }
            }
        },
        [],
    );

    if (articleSlug !== articleSlug.toLowerCase()) {
        return (
            <RedirectWithStatus
                to={internalUrl(
                    generatePath(URLs.journalArticle, {
                        articleSlug: articleSlug.toLowerCase(),
                    }),
                    query,
                    {trailingSlash: true},
                )}
                statusCode={ERedirectStatusCodes.PERMANENTLY}
            />
        );
    }

    // TODO: Ошибка запроса данных не должна интерпретироваться как 404
    if (!isFetched) {
        return null;
    } else if (!data) {
        return <NotFoundApp />;
    }

    const {pageTitle, articleImg, pageContent, author, createdDate} = data;

    return (
        <ContentLayout
            imageSrc={articleImg}
            imageAlt={pageTitle}
            title={pageTitle}
            navigation={buildNavigation(pageContent)}
        >
            <JournalPageMeta
                seo={data.seo}
                canonicalUrl={internalUrl(
                    generatePath(URLs.journalArticle, {articleSlug}),
                    null,
                    {withOrigin: true, trailingSlash: true},
                )}
            />

            <ArticleMeta article={data} />

            <Box
                below={deviceType.isDesktop ? 5 : 1}
                between={deviceType.isDesktop ? 8 : 6}
            >
                <Tags tags={tags} />

                <Flex
                    justifyContent="space-between"
                    flexDirection={deviceType.isDesktop ? 'row' : 'column'}
                    between={deviceType.isDesktop ? 1 : 4}
                >
                    <Author author={author} />

                    <Text
                        className={cx(deviceMods('date', deviceType))}
                        color="secondary"
                    >
                        {formatDate(
                            moment(createdDate, HUMAN_DATE_RU),
                            HUMAN_WITH_YEAR,
                            {withNbsp: true},
                        )}
                    </Text>
                </Flex>
            </Box>

            <Box>
                {pageContent.map((item, index) => {
                    const refForLastBlock =
                        index === pageContent.length - 1 ? ref : undefined;

                    if (isMarkupBlock(item)) {
                        return (
                            <MarkupBlock
                                className={cx(
                                    'markup',
                                    deviceMods('markup', deviceType),
                                    {
                                        markup_withTitle: Boolean(
                                            item.blockTitle,
                                        ),
                                    },
                                )}
                                key={index}
                                title={item.blockTitle}
                                navigationTitle={item.navigationTitle}
                                content={item.content}
                                ref={refForLastBlock}
                                onLinkClick={onLinkClick}
                            />
                        );
                    }

                    if (isImageBlock(item)) {
                        return (
                            <ImageBlock
                                className={cx(
                                    'image',
                                    deviceMods('image', deviceType),
                                )}
                                key={index}
                                title={item.blockTitle}
                                navigationTitle={item.navigationTitle}
                                images={item.imgSet}
                                ref={refForLastBlock}
                            />
                        );
                    }

                    if (isMapBlock(item)) {
                        return (
                            <MapBlock
                                className={cx(
                                    'map',
                                    deviceMods('map', deviceType),
                                )}
                                key={index}
                                id={item.content}
                                title={item.blockTitle}
                                navigationTitle={item.navigationTitle}
                                ref={refForLastBlock}
                            />
                        );
                    }

                    if (isCTALinkBlock(item)) {
                        return (
                            <CTALink
                                className={cx(
                                    'ctaLink',
                                    deviceMods('ctaLink', deviceType),
                                )}
                                key={index}
                                text={item.linkText}
                                url={item.linkUrl}
                                ref={refForLastBlock}
                                onLinkClick={onLinkClick}
                            />
                        );
                    }

                    return null;
                })}

                {author.type === EAuthorType.ZEN && author.url && (
                    <ZenAuthorFooter
                        className={cx(deviceMods('zenFooter', deviceType))}
                        link={author.url}
                    />
                )}
            </Box>
        </ContentLayout>
    );
};

export default Article;
