import React, {useMemo, ComponentType} from 'react';

import {IIconProps} from 'icons/types/icon';
import {IWithClassName} from 'types/withClassName';
import {
    EAuthorType,
    IContextPageAuthor,
} from 'server/api/TemplatorApi/types/IGetContextPageResponse';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Avatar from 'components/Avatar/Avatar';
import ZenS from 'icons/illustrations/Zen/ZenS';
import TravelS from 'icons/illustrations/Travel/TravelS';

import cx from './Author.scss';

const ICON_BY_TYPE: Record<EAuthorType, ComponentType<IIconProps> | undefined> =
    {
        [EAuthorType.ZEN]: ZenS,
        [EAuthorType.TRAVEL]: TravelS,
        [EAuthorType.TYPICAL]: undefined,
    };

export interface IAuthorProps extends IWithClassName, IWithQaAttributes {
    author: IContextPageAuthor;
}

const Author: React.FC<IAuthorProps> = props => {
    const deviceType = useDeviceType();
    const {
        className,
        author: {name, avatar, description, type, url},
    } = props;

    const descriptionContent = useMemo(() => {
        const Icon = type ? ICON_BY_TYPE[type] : undefined;

        return (
            <Flex alignItems="center" className={cx('description')}>
                {Icon && (
                    <Icon
                        className={cx('descriptionIcon')}
                        width={14}
                        height={14}
                    />
                )}
                <span>{description}</span>
            </Flex>
        );
    }, [description, type]);

    return (
        <Flex
            className={cx('root', className)}
            inline
            between={2}
            alignItems="flex-start"
            nowrap
        >
            <Avatar className={cx('avatar')} src={avatar} size="m" />

            <Flex flexDirection="column">
                <Text
                    className={cx('author', deviceMods('author', deviceType))}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'authorName',
                    })}
                >
                    {name}
                </Text>

                {url ? (
                    <Link url={url} target="_blank" rel="noopener noreferrer">
                        {descriptionContent}
                    </Link>
                ) : (
                    descriptionContent
                )}
            </Flex>
        </Flex>
    );
};

export default Author;
