import React, {useCallback, useMemo, useRef, useState} from 'react';
import {noop} from 'lodash';

import {IWithClassName} from 'types/withClassName';
import {IJournalArticleImg} from 'projects/journal/types/article/IJournalArticleImg';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useUniversalLayoutEffect} from 'utilities/hooks/useUniversalLayoutEffect';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import TravelImage from 'components/TravelImage/TravelImage';
import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';
import ImagesSwipeCarousel from 'components/ImagesSwipeCarousel/ImagesSwipeCarousel';

import ImageCarouselController, {
    IRenderImageCarouselControllerProps,
} from '../ImageCarouselController/ImageCarouselController';

import cx from './ImageCarouselBlock.scss';

interface IImageCarouselBlockProps extends IWithClassName {
    title?: string;
    images: IJournalArticleImg[];
}

const ImageCarouselBlock: React.FC<IImageCarouselBlockProps> = props => {
    const {className, images, title} = props;
    const deviceType = useDeviceType();
    const imageRef = useRef<HTMLImageElement>(null);
    const [imageMounted, setImageMounted] = useState<boolean>(false);
    const [carouselStyle, setCarouselStyle] = useState<React.CSSProperties>();
    const carouselImages = useMemo(
        () => images.map(({img}) => ({src: img})),
        [images],
    );

    const handleImageLoad = useCallback(() => {
        setImageMounted(true);
    }, [setImageMounted]);

    useUniversalLayoutEffect(() => {
        if (imageMounted && imageRef.current) {
            setCarouselStyle({
                height: `${imageRef.current.offsetHeight}px`,
            });
        }
    }, [imageRef, imageMounted]);

    return (
        <Box className={className} between={deviceType.isDesktop ? 5 : 3}>
            {Boolean(title) && (
                <Heading
                    className={cx('title', deviceMods('title', deviceType))}
                    level={3}
                >
                    {title}
                </Heading>
            )}

            {deviceType.isDesktop ? (
                <ImageCarouselController images={images}>
                    {({
                        onNextButtonClick,
                        onPrevButtonClick,
                    }: IRenderImageCarouselControllerProps): React.ReactElement => (
                        <ImagesCarousel
                            className={cx('imageCarouselWrapper')}
                            imageWrapClassName={cx(
                                'carousel',
                                'carousel_desktop',
                            )}
                            itemClassName={cx('carouselImage')}
                            withoutImageClassName={cx('withoutImage')}
                            images={carouselImages}
                            type="mini"
                            isDesktop={deviceType.isDesktop}
                            onNextButtonClick={onNextButtonClick}
                            onPrevButtonClick={onPrevButtonClick}
                            isLoadImageOnViewportIntersect
                        />
                    )}
                </ImageCarouselController>
            ) : (
                <div className={cx('wrapper')}>
                    <TravelImage
                        className={cx('heightKeeper')}
                        imageRef={imageRef}
                        src={images[0].img}
                        imageAlt={images[0].comment}
                        isWide
                        isLoadOnViewportIntersect
                        onLoad={handleImageLoad}
                    />

                    <ImageCarouselController images={images}>
                        {({
                            onScrollContent,
                        }: IRenderImageCarouselControllerProps): React.ReactElement => (
                            <ImagesSwipeCarousel
                                borderRadius="m"
                                borderWidth="s"
                                images={carouselImages}
                                onImageClick={noop}
                                swipeableStyle={carouselStyle}
                                onScrollContent={onScrollContent}
                                swipeableClassName={cx(
                                    'carousel',
                                    'carousel_mobile',
                                )}
                            />
                        )}
                    </ImageCarouselController>
                </div>
            )}
        </Box>
    );
};

export default ImageCarouselBlock;
