import React from 'react';
import {useInView} from 'react-intersection-observer';

import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Box from 'components/Box/Box';
import YandexConstructorMap from 'components/YandexConstructorMap/YandexConstructorMap';
import ArticleBlockTitle from 'projects/journal/components/ArticleBlockTitle/ArticleBlockTitle';

import cx from './MapBlock.scss';

interface IMapBlockProps extends IWithClassName {
    id: string;
    title?: string;
    navigationTitle?: string;
}

const DESKTOP_MAP_HEIGHT = '480';
const MOBILE_MAP_HEIGHT = '66vw';

const MapBlock = React.forwardRef<HTMLDivElement, IMapBlockProps>(
    (props, ref) => {
        const {className, title, id, navigationTitle} = props;
        const deviceType = useDeviceType();
        const mapHeight = deviceType.isDesktop
            ? DESKTOP_MAP_HEIGHT
            : MOBILE_MAP_HEIGHT;
        const [mapRef, inView] = useInView({
            triggerOnce: true,
            rootMargin: '0px 0px 600px 0px',
        });

        return (
            <Box
                className={className}
                between={deviceType.isDesktop ? 5 : 3}
                ref={mapRef}
            >
                {title && (
                    <ArticleBlockTitle
                        title={title}
                        anchor={navigationTitle}
                        level={3}
                    />
                )}

                <div
                    className={cx(
                        'container',
                        deviceMods('container', deviceType),
                    )}
                    ref={ref}
                >
                    {inView && (
                        <YandexConstructorMap um={id} height={mapHeight} />
                    )}
                </div>
            </Box>
        );
    },
);

MapBlock.displayName = 'MapBlock';

export default MapBlock;
