import React from 'react';

import {TMarkupItem} from 'projects/journal/types/markup';
import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {buildMarkup} from 'projects/journal/utilities/markup/buildMarkup';

import Box from 'components/Box/Box';
import ArticleBlockTitle from 'projects/journal/components/ArticleBlockTitle/ArticleBlockTitle';

import cx from './MarkupBlock.scss';

interface IMarkupBlockProps extends IWithClassName {
    title?: string;
    navigationTitle?: string;
    content?: TMarkupItem;
    onLinkClick?: (
        e: React.MouseEvent<HTMLDivElement, MouseEvent> & {
            target: HTMLAnchorElement;
        },
    ) => void;
}

const MarkupBlock = React.forwardRef<HTMLElement, IMarkupBlockProps>(
    (props, ref) => {
        const {title, navigationTitle, content, onLinkClick, className} = props;
        const deviceType = useDeviceType();

        return (
            <Box
                className={cx(
                    'root',
                    className,
                    deviceMods('root', deviceType),
                )}
                between={deviceType.isDesktop ? 5 : 3}
                onClick={onLinkClick}
                ref={ref}
            >
                {title && (
                    <ArticleBlockTitle
                        className={cx('header')}
                        title={title}
                        anchor={navigationTitle}
                        level={2}
                    />
                )}

                {content && (
                    <div
                        className={cx(
                            'content',
                            deviceMods('content', deviceType),
                        )}
                    >
                        {buildMarkup(content)}
                    </div>
                )}
            </Box>
        );
    },
);

MarkupBlock.displayName = 'MarkupBlock';

export default MarkupBlock;
