import React, {ReactNode, useCallback, useState} from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {buildMarkup} from 'projects/journal/utilities/markup/buildMarkup';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Form from 'components/Form/Form';
import Button from 'components/Button/Button';
import TextArea from 'components/TextArea/TextArea';
import Separator from 'components/Separator/Separator';
import FormField from 'components/FormField/FormField';

import cx from './MarkupViewer.scss';

enum EFieldName {
    TEXT = 'text',
}

interface IMarkupFormValues {
    [EFieldName.TEXT]: string;
}

const INITIAL_FORM_VALUES: IMarkupFormValues = {
    [EFieldName.TEXT]: '',
};

const ContentMarkupViewer: React.FC = () => {
    const deviceType = useDeviceType();
    const [markup, setMarkup] = useState<React.ReactNode>(null);

    const validateMarkup = useCallback((values: IMarkupFormValues) => {
        try {
            const jsonData = JSON.parse(values[EFieldName.TEXT]);

            try {
                buildMarkup(jsonData);
            } catch (err) {
                return {
                    [EFieldName.TEXT]:
                        'Возникла ошибка при создании вёрстки - скорей всего, проблема в лапках разработчика',
                };
            }
        } catch (err) {
            return {
                [EFieldName.TEXT]:
                    'Текст не является валидным JSON объектом - проверьте ввод',
            };
        }
    }, []);

    const handleFormSubmit = useCallback(
        (values: IMarkupFormValues) => {
            setMarkup(buildMarkup(JSON.parse(values[EFieldName.TEXT])));
        },
        [setMarkup],
    );

    return (
        <Form
            onSubmit={handleFormSubmit}
            validate={validateMarkup}
            initialValues={INITIAL_FORM_VALUES}
        >
            {({handleSubmit}): ReactNode => (
                <form onSubmit={handleSubmit} className={cx('form')}>
                    <Flex
                        flexDirection="column"
                        justifyContent="center"
                        alignItems="center"
                        y={8}
                    >
                        <Box className={cx('input')} between={4}>
                            <FormField
                                name={EFieldName.TEXT}
                                deviceType={deviceType}
                                control={(
                                    {input},
                                    {error, controlRef},
                                ): ReactNode => (
                                    <TextArea
                                        className={cx('text')}
                                        rows={10}
                                        value={input.value}
                                        onChange={input.onChange}
                                        state={error ? 'error' : undefined}
                                        placeholder="Вёрстка в JSON"
                                        wrapRef={controlRef}
                                    />
                                )}
                            />

                            <Button
                                className={cx('submit')}
                                theme="primary"
                                width="max"
                                size="l"
                                type="submit"
                            >
                                {i18nBlock.submit()}
                            </Button>
                        </Box>

                        <Separator margin={4} />

                        <Box className={cx('markup')}>{markup}</Box>
                    </Flex>
                </form>
            )}
        </Form>
    );
};

export default ContentMarkupViewer;
