import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';
import {isString} from 'lodash';

import {URLs} from 'constants/urls';

import journalRootPageSelector from 'selectors/journal/journalRootPageSelector';
import journalTagsListSelector from 'selectors/journal/journalTagsListSelector';

import {internalUrl} from 'utilities/url';
import {deviceMods} from 'utilities/stylesUtils';
import {mapTags} from 'projects/journal/utilities/mapTags';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useArticles} from 'projects/journal/utilities/useArticles';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';
import {useFetchRootPage} from 'projects/journal/pages/Root/utilities/useFetchRootPage';

import * as i18nBlock from 'i18n/journal';

import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import TagsList from 'projects/journal/components/TagsList/TagsList';
import CommonLayout from 'projects/journal/components/CommonLayout/CommonLayout';
import Articles from 'projects/journal/components/Articles/Articles';
import HeaderCard from 'projects/journal/components/HeaderCard/HeaderCard';
import JournalPageMeta from 'projects/journal/components/JournalPageMeta/JournalPageMeta';

import cx from './Root.scss';

const ITEMS_PER_PAGE = 8;

const Root: React.FC = () => {
    useFetchRootPage();

    const deviceType = useDeviceType();
    const query = useQueryByLocation();
    const page =
        query.page && isString(query.page) ? parseInt(query.page, 10) : 1;
    const {data, isFetched} = useSelector(journalRootPageSelector);
    const {data: tags} = useSelector(journalTagsListSelector);
    const meta = data?.meta;
    const articles = useArticles();

    const articlesToPaginate = useMemo(() => {
        if (!articles) {
            return [];
        }

        return deviceType.isDesktop ? articles.slice(1) : articles;
    }, [articles, deviceType]);
    const headerArticle = deviceType.isDesktop && page === 1 && articles?.[0];

    if (!isFetched) {
        return null;
    }

    return (
        <>
            <CommonLayout>
                <JournalPageMeta
                    seo={meta?.seo}
                    canonicalUrl={internalUrl(URLs.journalRoot, null, {
                        withOrigin: true,
                        trailingSlash: true,
                    })}
                />
                <Flex
                    className={cx('content', deviceMods('content', deviceType))}
                    alignItems={deviceType.isDesktop ? 'center' : 'flex-start'}
                    flexDirection="column"
                    above="10"
                >
                    <Heading
                        level={1}
                        className={cx('title', deviceMods('title', deviceType))}
                    >
                        {meta?.title || i18nBlock.articlesListTitle()}
                    </Heading>

                    {headerArticle && (
                        <HeaderCard
                            semanticId={headerArticle.semanticId}
                            imageSrc={headerArticle.articleImg}
                            title={headerArticle.pageTitle}
                            author={headerArticle.author}
                            publishDate={headerArticle.createdDate}
                            className={cx('firstArticle')}
                            tags={mapTags(headerArticle.tags, tags)}
                        />
                    )}

                    <Articles
                        articles={articlesToPaginate}
                        articlesPerPage={ITEMS_PER_PAGE}
                        additionalContent={
                            tags && Boolean(tags.length)
                                ? {
                                      content: (
                                          <TagsList
                                              className={cx(
                                                  'tagsList',
                                                  deviceMods(
                                                      'tagsList',
                                                      deviceType,
                                                  ),
                                              )}
                                              tags={tags}
                                          />
                                      ),
                                      insertIndex: 2,
                                  }
                                : undefined
                        }
                        className={cx('articles')}
                        tags={tags}
                    />
                </Flex>
            </CommonLayout>
        </>
    );
};

export default Root;
