import React, {useMemo} from 'react';
import {useParams, generatePath} from 'react-router-dom';
import {useSelector} from 'react-redux';
import {isString} from 'lodash';

import {URLs} from 'constants/urls';

import journalTagPageSelector from 'selectors/journal/journalTagPageSelector';

import {internalUrl} from 'utilities/url';
import {deviceMods} from 'utilities/stylesUtils';
import {useFetchTagPage} from './utilities/useFetchTagPage';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useArticles} from 'projects/journal/utilities/useArticles';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import Articles from 'projects/journal/components/Articles/Articles';
import NotFoundPage from 'components/NotFound/NotFoundPage/NotFoundPage';
import HeaderCard from 'projects/journal/components/HeaderCard/HeaderCard';
import CommonLayout from 'projects/journal/components/CommonLayout/CommonLayout';
import JournalPageMeta from 'projects/journal/components/JournalPageMeta/JournalPageMeta';

import cx from './Tag.scss';

const ITEMS_PER_PAGE = 2;

const Tag: React.FC = () => {
    const {tagSlug} = useParams<{tagSlug: string}>();

    useFetchTagPage(tagSlug);

    const {data, isFetched} = useSelector(journalTagPageSelector);
    const deviceType = useDeviceType();
    const query = useQueryByLocation();
    const page =
        query.page && isString(query.page) ? parseInt(query.page, 10) : 1;
    const meta = data?.meta;
    const articles = useArticles();

    const articlesToPaginate = useMemo(() => {
        if (!articles) {
            return [];
        }

        return deviceType.isDesktop ? articles.slice(1) : articles;
    }, [articles, deviceType]);
    const headerArticle = deviceType.isDesktop && page === 1 && articles?.[0];

    if (!isFetched) {
        return null;
    } else if (!meta) {
        return <NotFoundPage />;
    }

    return (
        <CommonLayout color={meta.color}>
            <JournalPageMeta
                seo={meta.seo}
                canonicalUrl={internalUrl(
                    generatePath(URLs.journalTag, {tagSlug}),
                    null,
                    {withOrigin: true, trailingSlash: true},
                )}
            />
            <Flex
                className={cx('content', deviceMods('content', deviceType))}
                alignItems={deviceType.isDesktop ? 'center' : 'flex-start'}
                flexDirection="column"
                above="10"
            >
                <Heading
                    level={1}
                    className={cx(
                        'tagTitle',
                        deviceMods('tagTitle', deviceType),
                    )}
                >
                    {`# ${meta.title}`}
                </Heading>
                <Text
                    className={cx(
                        'tagDescription',
                        deviceMods('tagDescription', deviceType),
                    )}
                >
                    {meta.tagDescription}
                </Text>
                {headerArticle && (
                    <HeaderCard
                        semanticId={headerArticle.semanticId}
                        imageSrc={headerArticle.articleImg}
                        title={headerArticle.pageTitle}
                        author={headerArticle.author}
                        publishDate={headerArticle.createdDate}
                        className={cx('firstArticle')}
                    />
                )}

                <Articles
                    articles={articlesToPaginate}
                    articlesPerPage={ITEMS_PER_PAGE}
                    className={cx('articles')}
                />
            </Flex>
        </CommonLayout>
    );
};

export default Tag;
