import React, {ReactNode, useCallback, useMemo} from 'react';
import {OnChange} from 'react-final-form-listeners';
import {useField, useForm} from 'react-final-form';

import {IWithClassName} from 'types/withClassName';
import {IAffiliatePlatform} from 'projects/partners/types/IAffiliatePlatform';
import {ELinkBuilderFormField} from 'projects/partners/pages/LinkBuilder/components/LinkBuilderForm/types/ELinkBuilderFormField';

import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBuilder from 'i18n/partnerLinkBuilder';

import Select from 'components/Select/Select';
import FormField from 'components/FormField/FormField';
import PartnerFieldLabel from 'projects/partners/pages/LinkBuilder/components/PartnerFieldLabel/PartnerFieldLabel';

interface IClidSelectProps extends IWithClassName {
    platforms: IAffiliatePlatform[];
}

const ClidSelect: React.FC<IClidSelectProps> = ({className, platforms}) => {
    const deviceType = useDeviceType();
    const formApi = useForm();
    const {
        input: {value: selectedPlatform},
    } = useField(ELinkBuilderFormField.PLATFORM);
    const clids = useMemo(
        () =>
            selectedPlatform
                ? platforms.find(({name}) => name === selectedPlatform)
                      ?.clids || []
                : [],
        [platforms, selectedPlatform],
    );
    const options = useMemo(
        () =>
            clids.map(item => ({
                value: item,
                data: item,
            })),
        [clids],
    );

    const handlePlatformChange = useCallback(() => {
        formApi.change(ELinkBuilderFormField.AFFILIATE_CLID, undefined);
    }, [formApi]);

    return (
        <>
            <FormField
                className={className}
                title={
                    <PartnerFieldLabel
                        label="CLID"
                        target={ELinkBuilderFormField.AFFILIATE_CLID}
                        hint={i18nBuilder.clidHint()}
                        useBottomSheet={false}
                    />
                }
                control={({input}): ReactNode => (
                    <Select<string, string>
                        {...input}
                        id={input.name}
                        options={options}
                        disabled={options.length === 0}
                        size="l"
                        width="max"
                        theme="outlined"
                        placeholder={CHAR_NBSP}
                    />
                )}
                initialValue={clids.length === 1 ? clids[0] : undefined}
                deviceType={deviceType}
                name={ELinkBuilderFormField.AFFILIATE_CLID}
            />

            <OnChange name={ELinkBuilderFormField.PLATFORM}>
                {handlePlatformChange}
            </OnChange>
        </>
    );
};

export default ClidSelect;
