import React, {useCallback, useEffect, useState} from 'react';
import {debounce, isEmpty} from 'lodash';
import {useFormState} from 'react-final-form';

import {EAffiliateQueryParams} from 'constants/affiliate/queryParams';
import {distributionUtmParams} from 'projects/partners/constants/partnersUtms';

import {ILinkBuilderForm} from 'projects/partners/pages/LinkBuilder/components/LinkBuilderForm/types/ILinkBuilderForm';

import {delay} from 'utilities/async/delay';
import {useMobile} from 'utilities/hooks/useMobile';
import {getPartnersLink} from 'projects/partners/utilities/urls/getPartnersLink';
import copyToClipboard from 'utilities/copyToClipboard';

import {copy} from 'i18n/common-share';
import * as i18nBuilder from 'i18n/partnerLinkBuilder';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';
import InputWithLoader, {
    EInputLoadingStatus,
} from 'components/InputWithLoader/InputWithLoader';

import {commonBrowserProvider} from 'serviceProvider/common/commonBrowserProvider';

import cx from './PartnerLink.scss';

const DEBOUNCE_TIMEOUT = 400;
const CLIPBOARD_HINT_TIMEOUT = 2000;

const PartnerLink: React.FC = () => {
    const isMobile = useMobile();
    const [error, setError] = useState<string>();
    const [state, setState] = useState<EInputLoadingStatus>(
        EInputLoadingStatus.DEFAULT,
    );
    const [partnerLink, setPartnerLink] = useState('');
    const {values: formValues, errors} = useFormState<ILinkBuilderForm>({
        subscription: {values: true, errors: true},
    });
    const [clipboardHint, setClipboardHint] = useState(false);
    const handleClipboardClick = useCallback(async () => {
        const success = await copyToClipboard({text: partnerLink});

        if (!success) {
            try {
                const target: HTMLInputElement | null =
                    document.querySelector('#partnerLink');

                if (!target) {
                    return;
                }

                target.select();
                document.execCommand('copy');
            } catch (e) {
                return;
            }
        }

        setClipboardHint(true);
        await delay(CLIPBOARD_HINT_TIMEOUT);
        setClipboardHint(false);
    }, [partnerLink]);

    const updatePartnerLink = useCallback(
        debounce(async (params: ILinkBuilderForm) => {
            if (!params.originalLink || !params.affiliateClid) {
                setPartnerLink('');
                setState(EInputLoadingStatus.DEFAULT);

                return;
            }

            try {
                setState(EInputLoadingStatus.LOADING);
                setPartnerLink(
                    await commonBrowserProvider.store({
                        url: getPartnersLink(
                            params.originalLink,
                            {
                                [EAffiliateQueryParams.AFFILIATE_CLID]:
                                    params.affiliateClid,
                                [EAffiliateQueryParams.AFFILIATE_VID]:
                                    params.affiliateVid,
                            },
                            distributionUtmParams,
                        ),
                    }),
                );
                setState(EInputLoadingStatus.DONE);
            } catch (e) {
                setError(i18nBuilder.linkBuildError());
                setPartnerLink('');
                setState(EInputLoadingStatus.DEFAULT);
            }
        }, DEBOUNCE_TIMEOUT),
        [],
    );

    useEffect((): void => {
        updatePartnerLink(isEmpty(errors) ? formValues : {});
    }, [formValues, errors, updatePartnerLink]);

    return (
        <>
            <Flex
                between="3"
                inline={!isMobile}
                flexDirection={isMobile ? 'column' : 'row'}
                nowrap
                below="4"
            >
                <InputWithLoader
                    id="partnerLink"
                    className={cx('partnerLink')}
                    loading={state}
                    value={partnerLink}
                    size="l"
                    onClick={partnerLink ? handleClipboardClick : undefined}
                    placeholder={i18nBuilder.partnerLinkPlaceholder()}
                />

                <Button
                    size="l"
                    disabled={!partnerLink}
                    onClick={handleClipboardClick}
                >
                    {copy()}
                </Button>
            </Flex>

            {error ? (
                <Text color="alert">{error}</Text>
            ) : (
                <Text
                    className={cx('clipboardHint', {
                        clipboardHint_visible: clipboardHint,
                    })}
                    color="success"
                >
                    {i18nBuilder.clipboardSuccess()}
                </Text>
            )}
        </>
    );
};

export default PartnerLink;
