import React, {ReactNode, useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IAffiliatePlatform} from 'projects/partners/types/IAffiliatePlatform';
import {ELinkBuilderFormField} from 'projects/partners/pages/LinkBuilder/components/LinkBuilderForm/types/ELinkBuilderFormField';

import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBuilder from 'i18n/partnerLinkBuilder';

import FormField from 'components/FormField/FormField';
import Select, {ISelectOption} from 'components/Select/Select';
import PartnerFieldLabel from 'projects/partners/pages/LinkBuilder/components/PartnerFieldLabel/PartnerFieldLabel';

interface IPlatformSelectProps extends IWithClassName {
    platforms: IAffiliatePlatform[];
}

const PlatformSelect: React.FC<IPlatformSelectProps> = ({platforms}) => {
    const deviceType = useDeviceType();
    const options = useMemo(
        () =>
            platforms.map(item => ({
                value: item.name,
                data: item,
            })),
        [platforms],
    );
    const renderItem = useCallback(
        (item: ISelectOption<string, IAffiliatePlatform>): string =>
            item.data.name,
        [],
    );

    return (
        <>
            <FormField
                title={
                    <PartnerFieldLabel
                        label={i18nBuilder.platformTitle()}
                        target={ELinkBuilderFormField.PLATFORM}
                    />
                }
                options={options}
                control={({input}): ReactNode => (
                    <Select<string, IAffiliatePlatform>
                        {...input}
                        id={input.name}
                        options={options}
                        disabled={options.length === 0}
                        size="l"
                        width="max"
                        theme="outlined"
                        renderItem={renderItem}
                        renderValue={renderItem}
                        placeholder={CHAR_NBSP}
                    />
                )}
                initialValue={options.length ? options[0].value : undefined}
                deviceType={deviceType}
                name={ELinkBuilderFormField.PLATFORM}
            />
        </>
    );
};

export default PlatformSelect;
