import React, {useCallback, useEffect} from 'react';
import {useParams} from 'react-router-dom';
import {isString} from 'lodash';

import {useAsync} from 'utilities/hooks/useAsync';

import {
    unsubscribePageTitle,
    unsubscribePageError,
    unsubscribePageErrorButton,
} from 'i18n/subscription';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Spinner from 'components/Spinner/Spinner';
import Heading from 'components/Heading/Heading';
import TicketCatM from 'icons/illustrations/TicketCat/TicketCatM';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';

import {notifierService} from 'serviceProvider';

import cx from './Unsubscribe.scss';

const UnsubscribePage: React.FC = () => {
    const {hash} = useParams<{hash?: string}>();
    const [unsubscribeStatus, unsubscribe] = useAsync(
        notifierService.provider()?.unsubscribe,
    );
    const unsubscribeByHash = useCallback(() => {
        if (isString(hash)) {
            unsubscribe(hash);
        }
    }, [hash, unsubscribe]);

    useEffect(() => {
        unsubscribeByHash();
    }, [unsubscribeByHash]);

    return (
        <LayoutDefault
            isFixedNavigation
            showSearchForm
            showHeader
            showNavigation
            hasSideSheetNavigation
        >
            <Flex
                className={cx('root')}
                alignItems="center"
                justifyContent="center"
                flexDirection="column"
                between="4"
                x="6"
                y="10"
            >
                {unsubscribeStatus.loading && <Spinner size="xxl" />}

                {unsubscribeStatus.error && (
                    <Flex
                        between="4"
                        alignItems="center"
                        flexDirection="column"
                    >
                        <Text>{unsubscribePageError()}</Text>

                        <Button onClick={unsubscribeByHash}>
                            {unsubscribePageErrorButton()}
                        </Button>
                    </Flex>
                )}

                {!unsubscribeStatus.error && !unsubscribeStatus.loading && (
                    <>
                        <TicketCatM width={130} height={130} />

                        <Heading className={cx('heading')} level={2}>
                            {unsubscribePageTitle()}
                        </Heading>
                    </>
                )}
            </Flex>
        </LayoutDefault>
    );
};

export default UnsubscribePage;
