import React from 'react';
import moment from 'moment';

import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import Header from 'projects/trains/components/OrderMainInfo/components/Header/Header';
import TrainCommonInfo from 'projects/trains/components/OrderMainInfo/components/TrainCommonInfo/TrainCommonInfo';
import TrainTimeAndStationsInfo from 'projects/trains/components/OrderMainInfo/components/TrainTimeAndStationsInfo/TrainTimeAndStationsInfo';

import cx from './OrderMainInfo.scss';

interface IOrderMainInfoProps {
    from: string;
    to: string;
    departure: string;
    departureTimezone: string;
    departureStation: string;
    arrival: string;
    arrivalTimezone: string;
    arrivalStation: string;
    startStation: string;
    endStation: string;
    coachNumber: string;
    coachType: ETrainsCoachType;
    places: string[];
    trainName: string;
    trainNumber: string;
    trainCompany?: string;
}

const ROOT_QA = 'orderTrainsInfo';

const OrderMainInfo: React.FC<IOrderMainInfoProps> = props => {
    const {
        from,
        to,
        departure,
        departureTimezone,
        arrival,
        arrivalTimezone,
        trainNumber,
        startStation,
        endStation,
        trainName,
        coachNumber,
        coachType,
        places,
        departureStation,
        arrivalStation,
        trainCompany,
    } = props;

    const deviceType = useDeviceType();

    const departureMoment = moment(departure).tz(departureTimezone);
    const arrivalMoment = moment(arrival).tz(arrivalTimezone);

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            <Header
                from={from}
                to={to}
                departureMoment={departureMoment}
                {...prepareQaAttributes(ROOT_QA)}
            />

            {deviceType.isMobile && <Separator className={cx('separator')} />}

            <Flex
                flexDirection={deviceType.isMobile ? 'column' : 'row'}
                above={deviceType.isMobile ? 4 : 1}
            >
                <TrainCommonInfo
                    className={cx('trainCommonInfo', cx('half'))}
                    trainNumber={trainNumber}
                    trainName={trainName}
                    trainCompany={trainCompany}
                    startStation={startStation}
                    endStation={endStation}
                    coachType={coachType}
                    coachNumber={coachNumber}
                    places={places}
                    {...prepareQaAttributes(ROOT_QA)}
                />

                <TrainTimeAndStationsInfo
                    className={cx('trainTimeAndStationsInfo', cx('half'))}
                    departureMoment={departureMoment}
                    departureStation={departureStation}
                    arrivalMoment={arrivalMoment}
                    arrivalStation={arrivalStation}
                    {...prepareQaAttributes(ROOT_QA)}
                />
            </Flex>
        </div>
    );
};

export default React.memo(OrderMainInfo);
