import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CAR_TYPE_UNKNOWN} from 'projects/account/lib/orders/trainsOrders';
import getCarTypeText from 'projects/trains/components/OrderMainInfo/components/TrainCommonInfo/utilities/getCarTypeText';

import * as i18nBlock from 'i18n/account-OrderTrains-Info';

import Box from 'components/Box/Box';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import Text from 'components/Text/Text';

interface ITrainCommonInfoProps extends IWithClassName, IWithQaAttributes {
    startStation: string;
    endStation: string;
    trainNumber: string;
    trainName: string;
    trainCompany?: string;
    coachNumber: string;
    coachType: ETrainsCoachType | 'unknown';
    places: string[];
}

const TrainCommonInfo: React.FC<ITrainCommonInfoProps> = props => {
    const {
        className,
        trainNumber,
        startStation,
        endStation,
        trainName,
        coachNumber,
        coachType,
        places,
        trainCompany,
    } = props;

    const deviceType = useDeviceType();

    return (
        <div className={className}>
            <Text
                weight={deviceType.isMobile ? 'bold' : 'normal'}
                size={deviceType.isMobile ? 'm' : 'xl'}
            >
                {trainCompany || i18nBlock.company()}
            </Text>

            <Box above={3}>
                <Text
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'mainInfo',
                    })}
                >
                    {i18nBlock.trainName({
                        trainNumber,
                        startStation,
                        endStation,
                    })}
                </Text>

                {trainName && (
                    <>
                        {deviceType.isMobile && <DotSeparator />}

                        <Text
                            color="highlight"
                            tag={deviceType.isMobile ? 'span' : 'div'}
                        >
                            {trainName}
                        </Text>
                    </>
                )}
            </Box>

            <Box above={deviceType.isMobile ? 3 : 0}>
                <Text {...prepareQaAttributes({parent: props, current: 'car'})}>
                    {i18nBlock.car({carNumber: coachNumber})}
                </Text>

                {coachType && coachType !== CAR_TYPE_UNKNOWN && (
                    <>
                        <DotSeparator />

                        <Text
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'carType',
                            })}
                        >
                            {getCarTypeText(coachType)}
                        </Text>
                    </>
                )}

                <DotSeparator />

                <Text
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'places',
                    })}
                >
                    {i18nBlock.places({
                        count: places.length,
                        places: places.join(', '),
                    })}
                </Text>
            </Box>
        </div>
    );
};

export default React.memo(TrainCommonInfo);
