import React from 'react';
import moment from 'moment';
import {Moment} from 'moment-timezone';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {formatDate} from 'utilities/dateUtils';
import {
    HUMAN_SHORT,
    HUMAN_SHORT_DATE_WITH_WEEK,
    TIME,
} from 'utilities/dateUtils/formats';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/account-OrderTrains-Info';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import Box from 'components/Box/Box';

import cx from './TrainTimeAndStations.scss';

interface ITrainTimeAndStationsInfoProps
    extends IWithClassName,
        IWithQaAttributes {
    departureMoment: Moment;
    arrivalMoment: Moment;
    departureStation: string;
    arrivalStation: string;
}

const TrainTimeAndStationsInfo: React.FC<ITrainTimeAndStationsInfoProps> =
    props => {
        const {
            className,
            departureMoment,
            arrivalMoment,
            departureStation,
            arrivalStation,
        } = props;

        const deviceType = useDeviceType();

        const duration = arrivalMoment.diff(departureMoment, 'hours', true);
        const durationHours = Math.floor(duration);
        const durationMinutes = Math.round((duration - durationHours) * 60);

        const arrivalDate = formatDate(
            arrivalMoment,
            HUMAN_SHORT_DATE_WITH_WEEK,
        );

        const departureStartOfDayMoment =
            moment(departureMoment).startOf('day');
        const arrivalStartOfDayMoment = moment(arrivalMoment).startOf('day');

        const diff = Math.round(
            arrivalStartOfDayMoment.diff(
                departureStartOfDayMoment,
                'days',
                true,
            ),
        );

        return (
            <div
                className={cx(
                    'root',
                    deviceMods('root', deviceType),
                    className,
                )}
            >
                <Flex
                    justifyContent="flex-end"
                    className={cx('localTimeAndArrivalDate')}
                >
                    {deviceType.isMobile && (
                        <Text
                            className={cx('localTime')}
                            color="alert"
                            size="s"
                        >
                            {i18nBlock.localDotMobile()}
                        </Text>
                    )}

                    <Text
                        className={cx('arrivalDate')}
                        color="secondary"
                        size="s"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrivalDate',
                        })}
                    >
                        {deviceType.isMobile
                            ? arrivalDate
                            : i18nBlock.dateTo({
                                  nextDay: diff === 1,
                                  date: arrivalStartOfDayMoment.format(
                                      HUMAN_SHORT,
                                  ),
                              })}
                    </Text>
                </Flex>

                <Flex alignItems="center">
                    <Text
                        className={cx('departureTime')}
                        weight="bold"
                        size="xl"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departureTime',
                        })}
                    >
                        {formatDate(departureMoment, TIME)}
                    </Text>

                    <Separator className={cx('timeSeparator')} />

                    <Box x={2} textSize={deviceType.isMobile ? 's' : 'm'}>
                        {i18nBlock.duration({
                            durationHours,
                            durationMinutes,
                        })}
                    </Box>

                    <Separator className={cx('timeSeparator')} />

                    <Text
                        className={cx('arrivalTime')}
                        weight="bold"
                        size="xl"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrivalTime',
                        })}
                    >
                        {formatDate(arrivalMoment, TIME)}
                    </Text>
                </Flex>

                <Flex
                    justifyContent="space-between"
                    above={deviceType.isMobile ? 0 : 3}
                >
                    <Text>{departureStation}</Text>

                    <Text className={cx('arrivalStation')}>
                        {arrivalStation}
                    </Text>
                </Flex>

                {deviceType.isDesktop && (
                    <Text color="alert">{i18nBlock.localDotDesktop()}</Text>
                )}
            </div>
        );
    };

export default React.memo(TrainTimeAndStationsInfo);
