import React, {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {
    TTrainsOrderStep,
    TTrainsSearchStep,
    ETrainsSearchStepType,
} from 'types/trains/common/steps/ITrainsSearchAndOrderSteps';

import changeOrderStep from 'reducers/trains/order/thunk/changeOrderStep';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import orderStepsSelector from 'projects/trains/components/OrderSteps/selectors/orderStepsSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import TrainsSearchAndOrderSteps from 'projects/trains/components/TrainsSearchAndOrderSteps/TrainsSearchAndOrderSteps';

import cx from './OrderSteps.scss';

interface ITrainsOrderStepsProps extends IWithQaAttributes {}

const ONE_WAY_SEARCH_STEPS: TTrainsSearchStep[] = [
    {
        type: ETrainsSearchStepType.ONE_WAY_FORWARD,
        isActive: false,
        disabled: false,
    },
];

const ROUND_TRIP_SEARCH_STEPS: TTrainsSearchStep[] = [
    {
        type: ETrainsSearchStepType.ROUND_TRIP_FORWARD,
        isActive: false,
        disabled: false,
    },
    {
        type: ETrainsSearchStepType.ROUND_TRIP_BACKWARD,
        isActive: false,
        disabled: false,
    },
];

const TrainsOrderSteps: React.FC<ITrainsOrderStepsProps> = props => {
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const context = useSelector(trainsContextSelector);
    const orderSteps = useSelector(orderStepsSelector);

    const handleChangeStep = useCallback(
        (step: TTrainsOrderStep) => {
            if (step.type === ORDER_STEP.PLACES) {
                dispatch(
                    changeOrderStep({
                        step: step.type,
                        direction: step.direction,
                        index: step.segmentIndex,
                    }),
                );
            } else {
                dispatch(changeOrderStep({step: step.type}));
            }
        },
        [dispatch],
    );

    if (!isFilledTrainsSearchContext(context)) {
        return null;
    }

    return (
        <TrainsSearchAndOrderSteps
            orderSteps={orderSteps}
            searchSteps={
                context.returnWhen
                    ? ROUND_TRIP_SEARCH_STEPS
                    : ONE_WAY_SEARCH_STEPS
            }
            x={deviceType.isMobile ? 4 : 5}
            breadcrumbsClassName={cx('root', deviceMods('root', deviceType))}
            handleChangeOrderStep={handleChangeStep}
            {...prepareQaAttributes(props)}
        />
    );
};

export default TrainsOrderSteps;
