import {createSelector} from 'reselect';
import flatMap from 'lodash/flatMap';
import times from 'lodash/times';

import {ORDER_STEP, ORDER_STEPS} from 'projects/trains/constants/orderSteps';

import {isEDirection} from 'types/common/EDirection';
import {TTrainsOrderStep} from 'types/trains/common/steps/ITrainsSearchAndOrderSteps';

import directionTrainsCountSelector from 'selectors/trains/order/directionTrainsCountSelector';
import segmentIndexSelector from 'selectors/trains/segmentIndexSelector';
import segmentDirectionSelector from 'selectors/trains/segmentDirectionSelector';
import availableStepsSelector from 'selectors/trains/order/availableStepsSelector';
import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';

const orderStepsSelector = createSelector(
    [
        directionTrainsCountSelector,
        orderStepSelector,
        availableStepsSelector,
        segmentDirectionSelector,
        segmentIndexSelector,
    ],
    (
        directionTrainsCount,
        currentStep,
        availableSteps,
        currentDirection,
        currentIndex,
    ): TTrainsOrderStep[] => {
        return flatMap(ORDER_STEPS, (step): TTrainsOrderStep[] => {
            if (step === ORDER_STEP.PLACES) {
                return flatMap(
                    Object.entries(directionTrainsCount),
                    ([direction, count]): TTrainsOrderStep[] => {
                        if (!isEDirection(direction)) {
                            throw new Error(
                                'trainsCount key is not EDirection',
                            );
                        }

                        return times(count).map(n => {
                            const stepDisabled = !availableSteps.find(
                                stepDescription =>
                                    stepDescription.step === step &&
                                    stepDescription.index === n &&
                                    stepDescription.direction === direction,
                            );

                            return {
                                type: step,
                                isActive:
                                    step === currentStep &&
                                    currentDirection === direction &&
                                    currentIndex === n,
                                disabled: stepDisabled,
                                direction,
                                segmentIndex: n,
                                segmentCount: count,
                            };
                        });
                    },
                );
            }

            const stepDisabled = !availableSteps.find(
                stepDescription => stepDescription.step === step,
            );

            return [
                {
                    type: step,
                    isActive: currentStep === step,
                    disabled: stepDisabled,
                },
            ];
        });
    },
);

export default orderStepsSelector;
