import React, {useRef, useMemo, useEffect} from 'react';
import {useSelector} from 'react-redux';

import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';

import {IWithClassName} from 'types/withClassName';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';

import orderSummarySelector from 'projects/trains/components/OrderSummary/selectors/orderSummarySelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18BlockSummary from 'i18n/trains-order-summary';

import {IButtonProps} from 'components/Button/Button';
import Skeleton from './components/Skeleton/Skeleton';
import Intersperse from 'components/Intersperse/Intersperse';
import Separator from 'components/Separator/Separator';
import TotalPrice from './components/TotalPrice/TotalPrice';
import GoToConfirmStep from './components/GoToConfirmStep/GoToConfirmStep';

import Insurance from './containers/InsuranceContainer/InsuranceContainer';
import TrainItem from './containers/TrainItemContainer/TrainItemContainer';
import TrainsOrderButton from '../TrainsOrderButton/TrainsOrderButton';

import cx from './OrderSummary.scss';

interface IOrderSummaryProps extends IWithClassName, IWithQaAttributes {
    hasGoToConfirmStep?: boolean;
    buttonRef?: React.Ref<HTMLButtonElement>;
}

const OrderSummary: React.FC<IOrderSummaryProps> = props => {
    const {className, hasGoToConfirmStep, buttonRef} = props;
    const {
        insuranceIsVisible,
        orderPageStatus,
        placesSteps,
        totalPrice,
        totalPriceIsVisible,
        trainItem,
        orderInfoState,
    } = useSelector(orderSummarySelector);

    const rootRef = useRef(null);
    const anchorRef = useRef<HTMLButtonElement | null>(null);

    useEffect(() => {
        if (typeof buttonRef === 'function' && anchorRef.current) {
            buttonRef(anchorRef.current);
        }
    }, [buttonRef]);

    const buttonProps = useMemo<IButtonProps>(
        () => ({
            innerRef: anchorRef,
        }),
        [],
    );

    if (
        orderPageStatus === ORDER_PAGE_STATUSES.FETCHING_DATA ||
        orderInfoState === EGenericOrderState.WAITING_RESERVATION
    ) {
        return <Skeleton />;
    }

    return (
        <div
            ref={rootRef}
            className={cx('root', className)}
            {...prepareQaAttributes(props)}
        >
            <div
                className={cx('title')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'title',
                })}
            >
                {i18BlockSummary.ticketsDashPrice()}
            </div>

            <Intersperse separator={<Separator margin={4} />}>
                {placesSteps.map(({direction, index}) => {
                    const key = `${direction}-${index}`;
                    const train = trainItem[direction][index];

                    if (!train?.isVisible) {
                        return null;
                    }

                    return (
                        <TrainItem
                            key={key}
                            className={cx('trainItem')}
                            direction={direction}
                            index={index}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'trainItem',
                                key,
                            })}
                        />
                    );
                })}
                {insuranceIsVisible && (
                    <Insurance
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'insurance',
                        })}
                    />
                )}
                {totalPriceIsVisible && (
                    <TotalPrice
                        className={cx('totalPrice')}
                        scopeRef={rootRef}
                        totalPriceWithoutDiscount={
                            totalPrice.refundableTotalPrice
                        }
                        {...totalPrice}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'totalPrice',
                        })}
                    />
                )}
            </Intersperse>

            <TrainsOrderButton
                className={cx('button')}
                buttonProps={buttonProps}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'orderButton',
                })}
            />
            {hasGoToConfirmStep && (
                <GoToConfirmStep
                    intersectionRef={anchorRef}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'goToConfirmStep',
                    })}
                />
            )}
        </div>
    );
};

export default React.memo(OrderSummary);
