import React, {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18n from 'i18n/trains-order-summary-details';

import Price from 'components/Price/Price';
import Checkbox from 'components/Checkbox/Checkbox';

import cx from './BeddingOption.scss';

interface IBeddingOptionProps extends IWithQaAttributes, IWithClassName {
    /**
     * Белье добавлено
     */
    checked: boolean;
    /**
     * Цена белья за одну штуку
     */
    price: IPrice;
    /**
     * Количество белья
     */
    count: number;
    /**
     * Только чтение
     */
    readOnly: boolean;

    onChange(checked: boolean): void;
}

const BeddingOption: React.FC<IBeddingOptionProps> = props => {
    const {price, count, readOnly, checked, onChange} = props;

    const handleChange = useCallback(
        e => {
            onChange(e.target.checked);
        },
        [onChange],
    );

    const text = useMemo(() => {
        if (count) {
            return i18n.beddingDashDetailsDashRedesign({count});
        }

        return i18n.beddingDashDetailsDashAdd({
            count,
            price: false,
        });
    }, [count]);

    return (
        <div className={cx('root')} {...prepareQaAttributes(props)}>
            {readOnly ? (
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'readOnly',
                    })}
                >
                    {text}
                </span>
            ) : (
                <>
                    <Checkbox
                        size="s"
                        checked={checked}
                        label={text}
                        onChange={handleChange}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'checkbox',
                        })}
                    />

                    {count !== 0 && (
                        <Price
                            {...price}
                            value={count * price.value}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'price',
                            })}
                        />
                    )}
                </>
            )}
        </div>
    );
};

export default BeddingOption;
