import {FC, useCallback, RefObject} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import orderSummarySelector from 'projects/trains/components/OrderSummary/selectors/orderSummarySelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {goToIntersectionNode} from './utilities/goToIntersectionNode';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/trains-order-confirm';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';

import {useIntersectionConfirmStepObserver} from './hooks/useIntersectionConfirmStepObserver';

import cx from './GoToConfirmStep.scss';

export interface IGoToConfirmStepProps
    extends IWithClassName,
        IWithQaAttributes {
    intersectionRef: RefObject<HTMLElement>;
}

// Возможно вынести в дальнейшем во что-то обобщенное на уровень проекта

const GoToConfirmStep: FC<IGoToConfirmStepProps> = props => {
    const {className, intersectionRef} = props;
    const deviceType = useDeviceType();
    const {totalPrice: totalPriceFullInfo, totalPriceIsVisible} =
        useSelector(orderSummarySelector);
    const totalPrice = totalPriceFullInfo?.totalPrice;
    const canSkipGoToAction =
        deviceType.isDesktop || !totalPrice || !totalPriceIsVisible;

    const {intersectionNode, hasIntersected} =
        useIntersectionConfirmStepObserver({
            canSkipGoToAction,
            intersectionRef,
        });

    const handleButtonClick = useCallback(() => {
        goToIntersectionNode(intersectionNode);
    }, [intersectionNode]);

    if (canSkipGoToAction || hasIntersected) {
        return null;
    }

    return (
        <div className={cx('root', className)} {...prepareQaAttributes(props)}>
            <Flex flexDirection="column" alignItems="center">
                <Text color="primary" size="s">
                    {insertJSXIntoKey(i18nBlock.goToConfirmPrice)({
                        priceNode: (
                            <Price
                                {...totalPrice}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'price',
                                })}
                            />
                        ),
                    })}
                </Text>
                <Button
                    className={cx('button')}
                    type="button"
                    theme="primary"
                    onClick={handleButtonClick}
                    size="l"
                    width="max"
                    {...prepareQaAttributes({parent: props, current: 'button'})}
                >
                    {i18nBlock.goToConfirmButtonLabel()}
                </Button>
            </Flex>
        </div>
    );
};

export default GoToConfirmStep;
