import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {reachGoalOnce} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';

import * as i18nBlock from 'i18n/trains-order-confirm';

import Price from 'components/Price/Price';
import Checkbox from 'components/Checkbox/Checkbox';
import Link from 'components/Link/Link';
import RenessansM from 'icons/illustrations/Renessans/RenessansM';

import cx from './Insurance.scss';

interface IInsuranceProps extends IWithClassName, IWithQaAttributes {
    disabled: boolean;
    price: IPrice;
    checked: boolean;
    onChange(checked: boolean): void;
}

const Insurance: React.FC<IInsuranceProps> = props => {
    const {className, disabled, price, checked, onChange} = props;

    const handleChange: React.ChangeEventHandler<HTMLInputElement> =
        useCallback(
            e => {
                onChange(e.target.checked);

                reachGoalOnce(ETrainsGoal.CLICK_INSURANCE_CHECKBOX);
            },
            [onChange],
        );

    const label = (
        <div className={cx('label')}>
            <div className={cx('textAndDescription')}>
                <div className={cx('text')}>
                    {i18nBlock.insuranceDashCheckboxDashTextDashRedesign()}
                </div>
                <div
                    className={cx('description')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'description',
                    })}
                >
                    {i18nBlock.insuranceDashDescriptionDashRedesign()}{' '}
                    <Link
                        className={cx('link')}
                        url={i18nBlock.insuranceDashLinkDashUrl()}
                        target="_blank"
                        theme="brand"
                    >
                        {i18nBlock.insuranceDashLinkDashTextDashRedesign()}
                    </Link>
                </div>
                <RenessansM
                    className={cx('logo')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'logo',
                    })}
                />
            </div>
            <Price
                className={cx('price')}
                {...price}
                {...prepareQaAttributes({parent: props, current: 'price'})}
            />
        </div>
    );

    return (
        <div className={cx('root', className)} {...prepareQaAttributes(props)}>
            <Checkbox
                {...prepareQaAttributes({parent: props, current: 'checkbox'})}
                className={cx('checkbox')}
                size="s"
                checked={checked}
                disabled={disabled}
                onChange={handleChange}
                label={label}
            />
        </div>
    );
};

export default Insurance;
