import React, {
    ChangeEventHandler,
    useCallback,
    useEffect,
    useMemo,
} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    ERefundableTariffRadioValue,
    isNonRefundableTariffToRadioValues,
} from './utilities/radioValuesToIsNonRefundableTariff';
import IPrice from 'utilities/currency/PriceInterface';
import {reachGoal} from 'utilities/metrika';
import getOptions from './utilities/getOptions';

import Radiobox from 'components/Radiobox/Radiobox';
import NonRefundableTariffRadioLabel from './components/NonRefundableTariffRadioLabel/NonRefundableTariffRadioLabel';

interface INonRefundableTariffProps extends IWithClassName {
    /**
     * Только чтение
     */
    readOnly: boolean;
    nonRefundablePlaces: number[];
    nonRefundablePrice: IPrice;
    refundablePrice: IPrice;
    discount: number;
    /**
     * Невозвратный тариф выбран
     */
    checked: boolean;
    hasRefundablePlace: boolean;
    onChange(checked: boolean): void;
}

const NonRefundableTariff: React.FC<INonRefundableTariffProps> = props => {
    const {
        className,
        checked,
        readOnly,
        discount,
        refundablePrice,
        hasRefundablePlace,
        nonRefundablePrice,
        nonRefundablePlaces,
        onChange,
    } = props;

    useEffect(() => {
        reachGoal(ETrainsGoal.NON_REFUNDABLE_TARIFF_SELECTOR_SHOWED);
    }, []);

    const handleChange: ChangeEventHandler<HTMLInputElement> = useCallback(
        e => {
            const isNonRefundableTariff =
                e.target.value ===
                ERefundableTariffRadioValue.NonRefundableTariff;

            reachGoal(ETrainsGoal.NON_REFUNDABLE_TARIFF_SELECTOR_CHANGED);

            onChange(isNonRefundableTariff);
        },
        [onChange],
    );

    const value = isNonRefundableTariffToRadioValues(checked);
    const options = useMemo(
        () =>
            getOptions({
                refundablePrice,
                nonRefundablePrice,
                nonRefundablePlaces,
                hasRefundablePlace,
                discount,
            }),
        [
            refundablePrice,
            nonRefundablePrice,
            nonRefundablePlaces,
            discount,
            hasRefundablePlace,
        ],
    );

    return (
        <div className={className}>
            {readOnly ? (
                <NonRefundableTariffRadioLabel
                    value={ERefundableTariffRadioValue.NonRefundableTariff}
                    hasRefundablePlace={hasRefundablePlace}
                    price={nonRefundablePrice}
                    places={nonRefundablePlaces}
                    discount={discount}
                    hidePrice
                />
            ) : (
                <Radiobox
                    size="s"
                    value={value}
                    options={options}
                    onChange={handleChange}
                />
            )}
        </div>
    );
};

export default NonRefundableTariff;
