import React, {useCallback, useMemo, RefObject} from 'react';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {REFUND_LINK} from 'projects/account/lib/orders/links';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {convertPlacesToString} from 'projects/trains/lib/order/places/convertPlacesToString';

import * as i18nBlock from 'i18n/trainsNonRefundableTariffSelector';

import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import cx from './NonRefundableTariffHelp.scss';

interface INonRefundableTariffHelpProps {
    places?: number[];
    isVisible: boolean;
    triggerRef: RefObject<HTMLElement>;
    discount: number;
    onClose: () => void;
}

const NonRefundableTariffHelp: React.FC<INonRefundableTariffHelpProps> =
    props => {
        const {places, onClose, isVisible, triggerRef, discount} = props;
        const deviceType = useDeviceType();
        const messageBoxProps = useMemo(() => {
            return {
                anchorRef: triggerRef,
                theme: EMessageBoxPopupTheme.WHITE,
                direction: EPopupDirection.BOTTOM_RIGHT,
            };
        }, [triggerRef]);

        const handleClose = useCallback(() => {
            onClose();
        }, [onClose]);

        return (
            <MessageBoxPopupOrBottomSheet
                isVisible={isVisible}
                onClose={handleClose}
                messageBoxProps={messageBoxProps}
            >
                <div className={cx(deviceMods('root', deviceType))}>
                    <Text size="l" tag="div">
                        {i18nBlock.nonRefundableTariffLabel()}
                    </Text>
                    <Text className={cx('description')} size="m" tag="div">
                        {insertJSXIntoKey(
                            i18nBlock.nonRefundableFullTariffHelp,
                        )({
                            discount: discount
                                ? i18nBlock.dynamicDiscount({discount})
                                : '',
                            placesInfo: convertPlacesToString({places}),
                            link: (
                                <Link
                                    url={REFUND_LINK}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                >
                                    {i18nBlock.helpLinkText()}
                                </Link>
                            ),
                        })}
                    </Text>
                </div>
            </MessageBoxPopupOrBottomSheet>
        );
    };

export default NonRefundableTariffHelp;
