import React, {useCallback, useRef} from 'react';

import IPrice from 'utilities/currency/PriceInterface';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {CHAR_SPACE} from 'utilities/strings/charCodes';
import {ERefundableTariffRadioValue} from '../../utilities/radioValuesToIsNonRefundableTariff';
import {convertPlacesToString} from 'projects/trains/lib/order/places/convertPlacesToString';

import * as i18nCommonBlock from 'i18n/common';
import * as i18nBlock from 'i18n/trainsNonRefundableTariffSelector';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import LinkButton from 'components/LinkButton/LinkButton';
import Label, {ECornersType, ELabelThemeType} from 'components/Label/Label';
import NonRefundableTariffHelp from 'projects/trains/components/TrainsOrderPage/NonRefundableTariffHelp/NonRefundableTariffHelp';

import cx from './NonRefundableTariffRadioLabel.scss';

interface INonRefundableTariffRadioLabelProps {
    value: ERefundableTariffRadioValue;
    price: IPrice;
    discount: number;
    places?: number[];
    hidePrice?: boolean;
    hasRefundablePlace: boolean;
}

const NonRefundableTariffRadioLabel: React.FC<INonRefundableTariffRadioLabelProps> =
    props => {
        const {
            price,
            places,
            hidePrice = false,
            discount,
            hasRefundablePlace,
        } = props;

        const {
            setTrue: onShow,
            setFalse: onClose,
            value: isVisible,
        } = useBoolean(false);

        const handleMoreClick: React.MouseEventHandler = useCallback(
            e => {
                onShow();

                e.preventDefault();
            },
            [onShow],
        );

        const triggerRef = useRef<HTMLDivElement | null>(null);

        return (
            <div>
                <Flex
                    className={cx('label')}
                    alignItems="center"
                    justifyContent="space-between"
                >
                    <Flex>
                        <Text size="s">
                            {i18nBlock.nonRefundableTariffLabel()}
                        </Text>
                        {!hidePrice && Boolean(discount) && (
                            <Label
                                className={cx('discount')}
                                size="s-inset"
                                cornersType={ECornersType.ROUNDED}
                                theme={ELabelThemeType.ATTENTION}
                            >
                                {i18nBlock.dynamicDiscount({discount})}
                            </Label>
                        )}
                    </Flex>
                    {!hidePrice && (
                        <Text size="s" weight="bold">
                            <Price
                                value={price.value}
                                currency={price.currency}
                            />
                        </Text>
                    )}
                </Flex>
                <Text
                    className={cx('description')}
                    size="xs"
                    tag="div"
                    color="secondary"
                >
                    {i18nBlock.nonRefundablePlacesDiscount({
                        placesInfo: convertPlacesToString({places}),
                    })}
                    .{CHAR_SPACE}
                    {hasRefundablePlace &&
                        `${i18nBlock.refundablePlacesWithoutDiscountLabel()}${CHAR_SPACE}`}
                    <LinkButton onClick={handleMoreClick} innerRef={triggerRef}>
                        {i18nCommonBlock.moreDashInfo()}
                    </LinkButton>
                </Text>
                <NonRefundableTariffHelp
                    places={places}
                    discount={discount}
                    onClose={onClose}
                    isVisible={isVisible}
                    triggerRef={triggerRef}
                    hasRefundablePlace={hasRefundablePlace}
                />
            </div>
        );
    };

export default NonRefundableTariffRadioLabel;
