import {FC, RefObject} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import IPrice from 'utilities/currency/PriceInterface';

import * as i18Block from 'i18n/trains-order-summary-details';

import Price from 'components/Price/Price';
import PriceExplanation from 'projects/trains/components/TrainsOrderPage/PriceExplanation/PriceExplanation';

import TotalPriceWithoutDiscount from '../TotalPriceWithoutDiscount/TotalPriceWithoutDiscount';

import cx from './TotalPrice.scss';

interface ITotalPriceProps extends IWithClassName, IWithQaAttributes {
    /**
     * Общая итоговая цена
     */
    totalPrice: IPrice;
    /**
     * Показать вопросик с детализацией итоговой цены
     */
    detailsIsVisible: boolean;
    /**
     * Цена сбора в итоговой цене
     */
    feePrice?: IPrice;
    /**
     * Цена билетов в итоговой цене
     */
    ticketsPrice?: IPrice;
    /**
     * Цена страховки в итоговой цене
     */
    insurancePrice?: IPrice;
    scopeRef?: RefObject<HTMLElement>;
    /**
     * Общая итоговая цена без скидки
     */
    totalPriceWithoutDiscount?: IPrice;
}

const TotalPrice: FC<ITotalPriceProps> = props => {
    const {
        className,
        totalPrice,
        scopeRef,
        detailsIsVisible,
        feePrice,
        ticketsPrice,
        insurancePrice,
        totalPriceWithoutDiscount,
    } = props;

    return (
        <footer
            className={cx('footer', className)}
            {...prepareQaAttributes(props)}
        >
            <span className={cx('label')}>
                {i18Block.detailsDashSummaryDashShort()}
            </span>
            <div className={cx('price')}>
                <TotalPriceWithoutDiscount
                    className={cx('totalPriceWithoutDiscount')}
                    totalPrice={totalPrice}
                    totalPriceWithoutDiscount={totalPriceWithoutDiscount}
                />
                <Price
                    className={cx('total')}
                    {...totalPrice}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'price',
                    })}
                />
                {detailsIsVisible && (
                    <PriceExplanation
                        className={cx('explanation')}
                        isMultipleTickets
                        ticketsPrice={ticketsPrice}
                        feePrice={feePrice}
                        insurancePrice={insurancePrice}
                        scopeRef={scopeRef}
                    />
                )}
            </div>
        </footer>
    );
};

export default TotalPrice;
