import {createSelector} from 'reselect';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';
import orderIsReservedSelector from 'selectors/trains/order/orderIsReservedSelector';
import orderPriceDetailsSelector from 'selectors/trains/order/orderPriceDetailsSelector';
import trainsSelector from 'selectors/trains/order/trainsSelector';
import orderInfoSelector from 'selectors/trains/order/orderInfoSelector';

import {getActualPlaces} from 'projects/trains/lib/order/places';
import IPrice from 'utilities/currency/PriceInterface';
import getBeddingIsIncluded from 'projects/trains/components/OrderSummary/utilities/getBeddingIsIncluded';
import {getIsShowBedding} from 'projects/trains/components/OrderSummary/utilities/getIsShowBedding';
import mapTrainsState, {
    TMapTrainsState,
} from 'projects/trains/lib/complexOrder/mapTrainsState';
import findTrainService from 'projects/trains/lib/complexOrder/findTrainService';

interface IBeddingStateProps {
    /**
     * Количество белья
     */
    count: number;
    /**
     * Цена белья за одну штуку
     */
    price: IPrice;
    /**
     * Только чтение
     */
    readOnly: boolean;
    /**
     * Белье включено в стоимость
     */
    included: boolean;
    /**
     * Белье добавлено
     */
    checked: boolean;
    /**
     * Отображается ли постельное белье
     */
    beddingIsVisible: boolean;
}

type TBeddingSelector = TMapTrainsState<IBeddingStateProps | undefined>;

const beddingContainerSelector = createSelector(
    [
        trainsSelector,
        orderStepSelector,
        orderInfoSelector,
        orderIsReservedSelector,
        orderPriceDetailsSelector,
    ],
    (
        trains,
        orderStep,
        orderInfo,
        isReserved,
        orderPriceDetails,
    ): TBeddingSelector => {
        return mapTrainsState(trains, (train, direction, index) => {
            const priceDetails = orderPriceDetails[direction][index];

            if (!train?.coach || !priceDetails) {
                return;
            }

            const service = findTrainService(orderInfo, direction, index);
            const passengers = service?.trainInfo?.passengers ?? [];
            const places = isReserved
                ? getActualPlaces(passengers)
                : train.orderPlaces;
            const {bedding} = priceDetails;
            const beddingIsVisible = getIsShowBedding(
                train?.coach,
                isReserved,
                train.beddingOption,
            );

            return {
                beddingIsVisible,
                checked: train.beddingOption,
                readOnly:
                    orderStep === ORDER_STEP.CONFIRM ||
                    orderStep === ORDER_STEP.PAYMENT,
                count: places.length,
                included: getBeddingIsIncluded(train.coach, isReserved),
                price: {value: bedding.single, currency: bedding.currency},
            };
        });
    },
);

export default beddingContainerSelector;
