import {createSelector} from 'reselect';

import trainsSelector from 'selectors/trains/order/trainsSelector';
import orderIsReservedSelector from 'selectors/trains/order/orderIsReservedSelector';
import orderPriceDetailsSelector from 'selectors/trains/order/orderPriceDetailsSelector';

import IPrice from 'utilities/currency/PriceInterface';
import mapTrainsState, {
    TMapTrainsState,
} from 'projects/trains/lib/complexOrder/mapTrainsState';

interface INonRefundableTariffStateProps {
    /**
     * Только чтение
     */
    readOnly: boolean;
    nonRefundablePlaces: number[];
    nonRefundablePrice: IPrice;
    refundablePrice: IPrice;
    /**
     * Невозвратный тариф выбран
     */
    checked: boolean;
    isVisible: boolean;
    discount: number;
    hasRefundablePlace: boolean;
}

type TNonRefundableTariffSelector = TMapTrainsState<
    INonRefundableTariffStateProps | undefined
>;

const nonRefundableTariffContainerSelector = createSelector(
    [trainsSelector, orderIsReservedSelector, orderPriceDetailsSelector],
    (trains, isReserved, orderPriceDetails): TNonRefundableTariffSelector => {
        return mapTrainsState(trains, (train, direction, index) => {
            const priceDetails = orderPriceDetails[direction][index];

            if (!priceDetails) {
                return;
            }

            const {
                nonRefundablePlaces,
                nonRefundablePrice,
                refundablePrice,
                coachPlaces,
            } = priceDetails;

            if (!nonRefundablePrice || !refundablePrice) {
                return;
            }

            const readOnly = isReserved;
            const discount = Math.round(
                100 - (nonRefundablePrice.value * 100) / refundablePrice.value,
            );

            return {
                checked: train.nonRefundableTariff,
                discount,
                readOnly,
                nonRefundablePlaces,
                nonRefundablePrice,
                refundablePrice,
                isVisible: Boolean(nonRefundablePlaces.length),
                hasRefundablePlace:
                    coachPlaces.length > nonRefundablePlaces.length,
            };
        });
    },
);

export default nonRefundableTariffContainerSelector;
