import React from 'react';

import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {convertPlacesToString} from 'projects/trains/lib/order/places/convertPlacesToString';

import * as i18n from 'i18n/trains-order-summary-details';

import Price from 'components/Price/Price';

import cx from './PlacesItem.scss';

interface IPlacesItemProps extends IWithClassName, IWithQaAttributes {
    /**
     * Название позиции (тариф)
     */
    title: string;
    /**
     * Выбранные места
     */
    places: number[];
    /**
     * Заглушка, когда мест нет
     */
    placesPlaceholder?: string;
    /**
     * Цена позиции
     */
    price?: IPrice;
    /**
     * Число позиций
     */
    totalPassengers: number;
}

const PlacesItem: React.FC<IPlacesItemProps> = props => {
    const {
        className,
        title,
        price,
        places,
        totalPassengers,
        placesPlaceholder = i18n.withoutDashSeat(),
    } = props;

    const placesCount = places.length;

    return (
        <li className={cx(className, 'root')} {...prepareQaAttributes(props)}>
            <span className={cx('description')}>
                <span
                    className={cx('title')}
                    {...prepareQaAttributes({parent: props, current: 'title'})}
                >
                    {title}
                </span>
                <span
                    className={cx('places')}
                    {...prepareQaAttributes({parent: props, current: 'places'})}
                >
                    {placesCount
                        ? convertPlacesToString({places, isLowerCase: true})
                        : placesPlaceholder}
                </span>
            </span>
            {totalPassengers > 1 && (
                <span className={cx('totalPassengers')}>
                    {totalPassengers}
                    {CHAR_NBSP}x{CHAR_NBSP}
                </span>
            )}
            {price && (
                <Price
                    {...price}
                    {...prepareQaAttributes({parent: props, current: 'price'})}
                />
            )}
        </li>
    );
};

export default PlacesItem;
