import {createSelector} from 'reselect';
import _flatMap from 'lodash/flatMap';
import _mapFp from 'lodash/fp/map';
import _groupFpBy from 'lodash/fp/groupBy';

import {isPassengersTypes} from 'projects/trains/constants/passengersTypes';

import {IPlace} from 'projects/trains/lib/order/getTrainPriceDetails/types/ITrainPriceDetails';

import trainsSelector from 'selectors/trains/order/trainsSelector';
import orderIsReservedSelector from 'selectors/trains/order/orderIsReservedSelector';
import orderPriceDetailsSelector from 'selectors/trains/order/orderPriceDetailsSelector';

import mapTrainsState, {
    TMapTrainsState,
} from 'projects/trains/lib/complexOrder/mapTrainsState';
import IPrice from 'utilities/currency/PriceInterface';
import getSummaryPassengerTypeText from 'projects/trains/lib/getSummaryPassengerTypeText';

import * as i18n from 'i18n/trains-order-summary-details';

interface IPlaceItemStateProps {
    /**
     * Название позиции (тариф)
     */
    title: string;
    /**
     * Выбранные места
     */
    places: number[];
    /**
     * Заглушка, когда мест нет
     */
    placesPlaceholder?: string;
    /**
     * Цена позиции
     */
    price?: IPrice;
    /**
     * Число позиций
     */
    totalPassengers: number;
}

type TPlacesContainerSelector = TMapTrainsState<IPlaceItemStateProps[]>;

const getPassengerTitleByType = (passengerType: string): string => {
    if (isPassengersTypes(passengerType)) {
        return getSummaryPassengerTypeText(passengerType);
    }

    return '';
};

const mergeGroupPlacesNumbers = (groupPlaces: IPlace[]): number[] =>
    _flatMap(groupPlaces, (place: IPlace): number[] => place.numbers);

const getGroupPlacesByPrice = _groupFpBy<IPlace>(place => place.price?.value);

const flatAndMapGroupPlaces = _mapFp<Record<number, IPlace[]>, IPlace>(
    (groupPlaces: IPlace[]) => {
        const [firstPlace] = groupPlaces;

        return {
            ...firstPlace,
            numbers: mergeGroupPlacesNumbers(groupPlaces),
        };
    },
);

const placesContainerSelector = createSelector(
    [trainsSelector, orderPriceDetailsSelector, orderIsReservedSelector],
    (trains, orderPriceDetails, isReserved): TPlacesContainerSelector => {
        return mapTrainsState(trains, (train, direction, index) => {
            const priceDetails = orderPriceDetails[direction][index];
            const placesWithPassengerType = Object.entries(priceDetails.places);

            return _flatMap(
                placesWithPassengerType,
                ([passengersType, places]) => {
                    const groupPlacesByPrice =
                        priceDetails.isPlacesSelectedAtOnce
                            ? null
                            : getGroupPlacesByPrice(places);

                    const placesByPrice = priceDetails.isPlacesSelectedAtOnce
                        ? places
                        : flatAndMapGroupPlaces(groupPlacesByPrice);

                    return placesByPrice.map(value => {
                        const {title, withoutPlace, numbers, price} = value;

                        const passengerTitleByType =
                            getPassengerTitleByType(passengersType);
                        const groupPlaces =
                            price && groupPlacesByPrice
                                ? groupPlacesByPrice[price.value]
                                : [];

                        return {
                            title: title || passengerTitleByType,
                            places: withoutPlace ? [] : numbers,
                            placesPlaceholder: isReserved
                                ? ''
                                : i18n.withoutDashSeat(),
                            price,
                            totalPassengers: groupPlaces?.length ?? 0,
                        };
                    });
                },
            );
        });
    },
);

export default placesContainerSelector;
