import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {EDirection} from 'types/common/EDirection';

import trainItemContainerSelector from 'projects/trains/components/OrderSummary/containers/TrainItemContainer/selectors/trainItemContainerSelector';
import beddingContainerSelector from 'projects/trains/components/OrderSummary/containers/BeddingContainer/selectors/beddingContainerSelector';
import nonRefundableTariffContainerSelector from 'projects/trains/components/OrderSummary/containers/NonRefundableTariffContainer/selectors/nonRefundableTariffContainerSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import PlacesPlaceholder, {
    EPlacesPlaceholderReason,
} from 'projects/trains/components/OrderSummary/components/PlacesPlaceholder/PlacesPlaceholder';
import Text from 'components/Text/Text';
import getFromToText from 'projects/trains/components/OrderSummary/containers/TrainItemContainer/utitlites/getFromToText';
import SpaceSeparator from 'components/SpaceSeparator/SpaceSeparator';

import Places from '../PlacesContainer/PlacesContainer';
import NonRefundableTariff from '../NonRefundableTariffContainer/NonRefundableTariffContainer';
import Bedding from '../BeddingContainer/BeddingContainer';

interface ITrainItemProps extends IWithClassName, IWithQaAttributes {
    direction: EDirection;
    index: number;
}

/**
 * Блок описывающий один поезд из заказа
 */
const TrainItemContainer: React.FC<ITrainItemProps> = props => {
    const {className, direction, index} = props;

    const bedding = useSelector(beddingContainerSelector);
    const trainItem = useSelector(trainItemContainerSelector);
    const nonRefundableTariff = useSelector(
        nonRefundableTariffContainerSelector,
    );
    const trainItemStateProps = trainItem[direction][index];

    if (!trainItemStateProps) {
        return null;
    }

    const currentBedding = bedding[direction][index];
    const currentNonRefundableTariff = nonRefundableTariff[direction][index];

    const {
        isMultipleTrains,
        settlementFrom,
        stationFrom,
        settlementTo,
        stationTo,
        noPlacesPlaceholderReason,
    } = trainItemStateProps;

    const {from, to} = getFromToText(
        {settlement: settlementFrom, station: stationFrom},
        {settlement: settlementTo, station: stationTo},
    );

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <Intersperse separator={<Separator margin={4} />}>
                <Intersperse
                    separator={
                        isMultipleTrains ? (
                            <SpaceSeparator space={2} />
                        ) : (
                            <Separator margin={4} />
                        )
                    }
                >
                    {isMultipleTrains && (
                        <Text
                            weight="bold"
                            size="s"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'title',
                            })}
                        >
                            {from} — {to}
                        </Text>
                    )}
                    {noPlacesPlaceholderReason === null ? (
                        <Places
                            direction={direction}
                            index={index}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'places',
                            })}
                        />
                    ) : (
                        <PlacesPlaceholder
                            reason={noPlacesPlaceholderReason}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'placesPlaceholder',
                            })}
                        />
                    )}
                    {noPlacesPlaceholderReason !==
                        EPlacesPlaceholderReason.NOT_SELECTED_PLACE &&
                        Boolean(currentBedding?.beddingIsVisible) && (
                            <Bedding
                                direction={direction}
                                index={index}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'bedding',
                                })}
                            />
                        )}
                </Intersperse>
                {currentNonRefundableTariff?.isVisible && (
                    <NonRefundableTariff direction={direction} index={index} />
                )}
            </Intersperse>
        </div>
    );
};

export default TrainItemContainer;
