import {createSelector} from 'reselect';

import {ITrainOrderState} from 'reducers/trains/order/types';
import {EPlacesView} from 'types/trains/booking/EPlacesView';

import trainsSelector from 'selectors/trains/order/trainsSelector';
import isMultipleTrainsSelector from 'selectors/trains/order/isMultipleTrainsSelector';
import orderIsReservedSelector from 'selectors/trains/order/orderIsReservedSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import mapTrainsState, {
    TMapTrainsState,
} from 'projects/trains/lib/complexOrder/mapTrainsState';
import compareDirectionAndIndex from 'projects/trains/lib/complexOrder/compareDirectionAndIndex';

import {EPlacesPlaceholderReason} from 'projects/trains/components/OrderSummary/components/PlacesPlaceholder/PlacesPlaceholder';

interface ITrainItemStateProps {
    isMultipleTrains: boolean;
    settlementFrom: string;
    stationFrom: string;
    settlementTo: string;
    stationTo: string;
    isVisible: boolean;
    noPlacesPlaceholderReason: EPlacesPlaceholderReason | null;
}

export type TTrainItemContainerSelector = TMapTrainsState<
    ITrainItemStateProps | undefined
>;

const trainItemContainerSelector = createSelector(
    [
        trainsSelector,
        isMultipleTrainsSelector,
        orderIsReservedSelector,
        currentSegmentDirectionAndIndexSelector,
    ],
    (
        trains,
        isMultipleTrains,
        isReserved,
        current,
    ): TTrainItemContainerSelector => {
        return mapTrainsState(trains, (train, direction, index) => {
            const {trainDetails: trainDetailsInfo, segment} = train;
            const trainDetails = trainDetailsInfo?.trainDetails;

            if (!trainDetails) {
                return;
            }

            const noPlacesPlaceholderReason = getNoPlacesPlaceholderReason(
                train,
                isReserved,
            );

            const isCurrentOrBeforeTrain =
                compareDirectionAndIndex({direction, index}, current) <= 0;

            return {
                isMultipleTrains,
                settlementFrom:
                    trainDetails.stationFrom?.settlement?.title ?? '',
                stationFrom:
                    (segment?.stationFrom.popularTitle ||
                        segment?.stationFrom.title) ??
                    '',
                settlementTo: trainDetails.stationTo?.settlement?.title ?? '',
                stationTo:
                    (segment?.stationTo.popularTitle ||
                        segment?.stationTo.title) ??
                    '',
                isVisible:
                    isCurrentOrBeforeTrain ||
                    noPlacesPlaceholderReason === null,
                noPlacesPlaceholderReason,
            };
        });
    },
);

export default trainItemContainerSelector;

function getNoPlacesPlaceholderReason(
    train: ITrainOrderState,
    isReserved: boolean,
): EPlacesPlaceholderReason | null {
    const {coach, placesViewType, orderPlaces} = train;
    const requirementsAreShown =
        (coach && !coach.schemaId) ||
        placesViewType === EPlacesView.REQUIREMENTS;

    if (isReserved) {
        return null;
    }

    if (requirementsAreShown) {
        return EPlacesPlaceholderReason.REQUIREMENTS;
    }

    if (!orderPlaces.length) {
        return EPlacesPlaceholderReason.NOT_SELECTED_PLACE;
    }

    return null;
}
