import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import currentOrderPlacesSelector from 'selectors/trains/order/currentOrderPlacesSelector';
import currentPriceDetailsSelector from 'selectors/trains/order/currentPriceDetailsSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';
import nonRefundableTariffContainerSelector from 'projects/trains/components/OrderSummary/containers/NonRefundableTariffContainer/selectors/nonRefundableTariffContainerSelector';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/common-plural';

import Info from 'icons/16/Info';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Flex, {IFlexProps} from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import OrderSummary from 'projects/trains/components/OrderSummary/OrderSummary';
import NonRefundableTariff from 'projects/trains/components/OrderSummary/containers/NonRefundableTariffContainer/NonRefundableTariffContainer';
import TotalPriceWithoutDiscount from 'projects/trains/components/OrderSummary/components/TotalPriceWithoutDiscount/TotalPriceWithoutDiscount';

import TrainsOrderButton from '../TrainsOrderButton/TrainsOrderButton';

import cx from './OrderSummaryCompact.scss';

interface IOrderSummaryCompactProps
    extends IWithClassName,
        IWithQaAttributes,
        Pick<IFlexProps, 'below' | 'above'> {
    buttonRef?: React.Ref<HTMLButtonElement>;
}

const OrderSummaryCompact: React.FC<IOrderSummaryCompactProps> = props => {
    const {className, below, above, buttonRef} = props;
    const orderPlaces = useSelector(currentOrderPlacesSelector);
    const {price, refundablePrice} = useSelector(currentPriceDetailsSelector);
    const nonRefundableTariff = useSelector(
        nonRefundableTariffContainerSelector,
    );
    const {direction, index} = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );

    const currentNonRefundableTariff = nonRefundableTariff[direction][index];

    const {value: isOpened, setTrue: open, setFalse: close} = useBoolean(false);

    if (!price?.value) {
        return null;
    }

    return (
        <Flex
            className={className}
            flexDirection="column"
            between={3}
            justifyContent="stretch"
            below={below}
            above={above}
            {...prepareQaAttributes(props)}
        >
            {currentNonRefundableTariff?.isVisible && (
                <>
                    <NonRefundableTariff direction={direction} index={index} />
                    <Separator margin={1} />
                </>
            )}
            <Flex inline justifyContent="space-between">
                <>
                    <Text
                        size="l"
                        weight="bold"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'places',
                        })}
                    >
                        {i18nBlock.places({count: orderPlaces.length})}
                    </Text>
                    <button
                        className={cx('button')}
                        type="button"
                        onClick={open}
                    >
                        <TextWithIcon
                            text={
                                <Flex inline alignItems="baseline">
                                    <TotalPriceWithoutDiscount
                                        className={cx(
                                            'totalPriceWithoutDiscount',
                                        )}
                                        size="s"
                                        totalPrice={price}
                                        totalPriceWithoutDiscount={
                                            refundablePrice
                                        }
                                    />
                                    <Text size="l" weight="bold">
                                        <Price
                                            className={cx('price')}
                                            {...price}
                                            {...prepareQaAttributes({
                                                parent: props,
                                                current: 'price',
                                            })}
                                        />
                                    </Text>
                                </Flex>
                            }
                            iconRight={Info}
                            iconRightClassName={cx('question')}
                        />
                    </button>
                    <BottomSheet
                        isOpened={isOpened}
                        onClose={close}
                        popHistoryOnUnmount={false}
                    >
                        <OrderSummary
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'orderSummary',
                            })}
                        />
                    </BottomSheet>
                </>
            </Flex>
            <TrainsOrderButton
                innerRef={buttonRef}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'orderButton',
                })}
            />
        </Flex>
    );
};

export default OrderSummaryCompact;
