import {FC, memo, useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    SOLOMON_TRAINS_PAYMENT_PAGE_LOADED,
    SOLOMON_TRAINS_TRUST_ERROR,
    SOLOMON_TRAINS_TRUST_LOADED,
    SOLOMON_TRAINS_TRUST_SUCCESS,
} from 'constants/solomon';

import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {setPaymentCompleted as setPaymentCompletedAction} from 'reducers/trains/order/actions/flow';

import trainsOrderSelector from 'selectors/trains/order/trainsOrderSelector';

import {getWaiterInfo} from 'projects/trains/lib/order/payment';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal, reachGoalOnce} from 'utilities/metrika';
import {sendPaymentCounter} from 'utilities/solomon';
import useQuery from 'utilities/hooks/useQuery';
import {usePrevious} from 'utilities/hooks/usePrevious';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';

import BookingMailSubscription from 'containers/BookingMailSubscription/BookingMailSubscription';

import {TrainsOrderMeta} from 'projects/trains/components/meta/TrainsOrderMeta/TrainsOrderMeta';
import OrderDetails from './components/OrderDetails/OrderDetails';
import PaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';
import PaymentIframe from 'components/PaymentIframe/PaymentIframe';
import useCancelOrderPollingIfNetworkGoesOffline from 'projects/trains/components/PaymentPage/hooks/useCancelOrderPollingIfNetworkGoesOffline';

const PAGE_QA = 'trainsPaymentPage';

interface IPaymentPageProps {}

const PaymentPage: FC<IPaymentPageProps> = () => {
    const [isLoaded, setIsLoaded] = useState(false);
    const [paymentIsStarted, setPaymentIsStarted] = useState(false);
    // при перезагрузке страницы из-за скрытия iframe'а траст считает,
    // что пользователь отменил оплату и показывает ошибку во фрейме, которую нам надо скрывать от пользователя
    const [isTrustIframeErrorVisible, setIsTrustIframeErrorVisible] =
        useState(false);

    const dispatch = useDispatch();
    const order = useSelector(trainsOrderSelector);

    const {orderInfo, paymentCompleted} = order;

    const deviceType = useDeviceType();

    const {id: orderUid} = useQuery(['id']);

    const prevPaymentUrl = usePrevious(orderInfo?.payment?.paymentUrl);
    const paymentUrl = orderInfo?.payment?.paymentUrl;

    const setPaymentCompleted = useDispatchedAction(
        setPaymentCompletedAction,
        dispatch,
    );

    const handlePaymentStarted = useCallback(() => {
        setPaymentIsStarted(true);
        setIsTrustIframeErrorVisible(false);
    }, []);

    const handlePaymentLoaded = useCallback(() => {
        reachGoalOnce(ETrainsGoal.PAYMENT_IFRAME_LOADED, {
            trains: {orderUid},
        });

        setIsTrustIframeErrorVisible(false);
        setIsLoaded(true);
    }, [orderUid]);

    const handlePaymentSuccess = useCallback(() => {
        setPaymentCompleted(true);

        setIsTrustIframeErrorVisible(false);
    }, [setPaymentCompleted]);

    const handlePaymentError = useCallback(() => {
        setIsTrustIframeErrorVisible(true);
    }, []);

    useEffect(() => {
        reachGoal(ETrainsGoal.ORDER_PAYMENT_PAGE_LOAD, {
            trains: {orderUid},
        });

        setPaymentCompleted(false);
        sendPaymentCounter(SOLOMON_TRAINS_PAYMENT_PAGE_LOADED);
    }, [orderUid, setPaymentCompleted]);

    useEffect(() => {
        if (paymentUrl !== prevPaymentUrl) {
            setIsLoaded(false);
            setIsTrustIframeErrorVisible(false);
            setPaymentCompleted(false);
        }
    }, [paymentUrl, prevPaymentUrl, setPaymentCompleted]);

    useCancelOrderPollingIfNetworkGoesOffline();

    const {waiterText, showWaiter} = getWaiterInfo({
        loaded: isLoaded,
        complete: paymentCompleted,
        isTrustIframeErrorVisible,
    });

    return (
        <>
            <TrainsOrderMeta />
            <PaymentLayout
                footer={
                    orderInfo && (
                        <BookingMailSubscription
                            email={orderInfo.contactInfo.email}
                            source={ESubscriptionSource.PAYMENT}
                            promoSubscriptionCode={ESubscriptionCode.travelNews}
                            travelVerticalName={
                                ESubscriptionVerticalName.Trains
                            }
                            subscriptionIsAvailable={paymentIsStarted}
                        />
                    )
                }
                orderInfo={
                    <OrderDetails
                        deviceType={deviceType}
                        order={order}
                        {...prepareQaAttributes({
                            parent: PAGE_QA,
                            current: 'orderDetails',
                        })}
                    />
                }
                loading={showWaiter}
                spinnerText={waiterText}
                {...prepareQaAttributes(PAGE_QA)}
            >
                {paymentUrl && (
                    <PaymentIframe
                        isNewTrust
                        loadEvent={SOLOMON_TRAINS_TRUST_LOADED}
                        successEvent={SOLOMON_TRAINS_TRUST_SUCCESS}
                        errorEvent={SOLOMON_TRAINS_TRUST_ERROR}
                        paymentUrl={paymentUrl}
                        onLoad={handlePaymentLoaded}
                        onStartPayment={handlePaymentStarted}
                        onPaymentSuccess={handlePaymentSuccess}
                        onPaymentError={handlePaymentError}
                        {...prepareQaAttributes({
                            parent: PAGE_QA,
                            current: 'iframe',
                        })}
                    />
                )}
            </PaymentLayout>
        </>
    );
};

export default memo(PaymentPage);
