import React, {Component} from 'react';
import moment from 'moment-timezone';
import upperFirst from 'lodash/upperFirst';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';

import {CHAR_NBSP, CHAR_EM_DASH} from 'utilities/strings/charCodes';
import {formatDate} from 'utilities/dateUtils';
import {HUMAN, TIME} from 'utilities/dateUtils/formats';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';
import humanizePeriod from 'utilities/dateUtils/humanizePeriod';
import formatPlacesInfo from 'projects/trains/lib/format/formatPlacesInfo';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nTrainsTransportTypesBlock from 'i18n/trains-transport-types';

import PaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';
import getFromToText from 'projects/trains/components/OrderSummary/containers/TrainItemContainer/utitlites/getFromToText';

interface IOrderDetailsProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    order: TrainsOrderStateInterface;
}

export default class OrderDetails extends Component<IOrderDetailsProps> {
    private getAdditionalInfo(trainServices: ITrainGenericService[]): string {
        if (trainServices.length === 1) {
            const {order} = this.props;

            return `${this.getDates(trainServices)}, ${formatPlacesInfo(
                trainServices[0],
                order,
            )}`;
        }

        return this.getDates(trainServices);
    }

    private getDates(trainServices: ITrainGenericService[]): string {
        const firstTainService = trainServices[0];
        const firstTainDepartureMoment = moment.tz(
            firstTainService.trainInfo.departure,
            firstTainService.trainInfo.stationFrom.timezone,
        );

        if (trainServices.length === 1) {
            return formatDate(
                firstTainDepartureMoment,
                `${HUMAN} ${TIME}`,
            ).toLowerCase();
        }

        const lastTainService = trainServices[trainServices.length - 1];
        const lastTainDepartureMoment = moment.tz(
            lastTainService.trainInfo.departure,
            lastTainService.trainInfo.stationFrom.timezone,
        );

        return humanizePeriod(
            firstTainDepartureMoment,
            lastTainDepartureMoment,
            {
                short: true,
            },
        );
    }

    private getDirections(trainServices: ITrainGenericService[]): string {
        const firstTainService = trainServices[0];
        const lastTainService = trainServices[trainServices.length - 1];

        const {stationFrom} = firstTainService.trainInfo;
        const {stationTo} = lastTainService.trainInfo;

        const {from, to} = getFromToText(
            {
                settlement: stationFrom.settlementTitle,
                station: stationFrom.popularTitle || stationFrom.title,
            },
            {
                settlement: stationTo.settlementTitle,
                station: stationTo.popularTitle || stationTo.title,
            },
        );
        const directions = `${from}${CHAR_NBSP}${CHAR_EM_DASH} ${to}`;

        if (trainServices.length === 1) {
            const {
                trainInfo: {trainTicketNumber},
            } = firstTainService.trainInfo;

            const train = `${upperFirst(
                i18nTrainsTransportTypesBlock.typeDashTrain(),
            )} ${trainTicketNumber}`;

            return `${train}, ${directions}`;
        }

        return directions;
    }

    render(): React.ReactNode {
        const {
            className,
            order: {orderInfo},
        } = this.props;

        if (!orderInfo) {
            return null;
        }

        const trainServices = getTrainServices(orderInfo);

        return (
            <PaymentLayout.PaymentOrderInfo
                className={className}
                descriptionGeneral={
                    <span
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'directions',
                        })}
                    >
                        {this.getDirections(trainServices)}
                    </span>
                }
                descriptionAdditional={
                    <span
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'additional',
                        })}
                    >
                        {this.getAdditionalInfo(trainServices)}
                    </span>
                }
                orderId={orderInfo?.prettyId}
                orderType={EProjectName.TRAINS}
            />
        );
    }
}
