import React, {useEffect, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import isEmpty from 'lodash/isEmpty';
import isEqual from 'lodash/isEqual';
import {ParsedQuery} from 'query-string';

import {RAILWAYS_SEARCH_FORM_NAME} from 'constants/reduxForm';
import {TLD} from 'constants/tld';

import {IWithClassName} from 'types/withClassName';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';
import {
    ITrainsSearchFormErrors,
    ITrainsSearchFormPointField,
    TTrainsSearchFormDateField,
} from 'projects/trains/components/SearchForm/types';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ITrainsSuggest} from 'types/trains/common/ITrainsApiSuggest';
import {ITrainsPreviousSearch} from 'types/trains/previousSearches/IPreviousSearch';
import {ECalendarType} from 'components/Calendar/types';

import {previousSearchFormSetFieldValue} from 'reducers/common/previousSearchForm/actions';
import {
    setTrainsSearchFormEndDateFieldAction,
    setTrainsSearchFormFromFieldAction,
    setTrainsSearchFormIsRoundTripFieldAction,
    setTrainsSearchFormStartDateFieldAction,
    setTrainsSearchFormToFieldAction,
} from 'reducers/trains/searchForm/actions';
import requestSearchSuggestsThunkAction from 'reducers/trains/searchSuggests/thunk/requestSearchSuggestsThunkAction';
import restorePreviousSearchesThunkAction from 'reducers/trains/previousSearches/thunk/restorePreviousSearchesThunkAction';
import fillSearchFormByPreviousSearchThunkAction from 'reducers/trains/previousSearches/thunk/fillSearchFormByPreviousSearch';
import addPreviousSearchThunkAction from 'reducers/trains/previousSearches/thunk/addPreviousSearchThunkAction';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import searchFormSelector from 'selectors/trains/searchForm/searchFormSelector';

import {ROBOT} from 'utilities/dateUtils/formats';
import history from 'utilities/browserHistory/browserHistory';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {trainsURLs} from 'projects/trains/lib/urls';
import validateForm from 'projects/trains/components/SearchForm/utilities/validateForm';
import getSuggestTitleAndDescription from 'projects/trains/components/SearchForm/utilities/getSuggestTitleAndDescription';
import {QueryInterface} from 'projects/trains/lib/urls/getTrainsSearchUrl';
import getWhenMoment from 'utilities/dateUtils/when/getWhenMoment';
import getPreviousFormValues from 'projects/trains/components/SearchForm/utilities/getPreviousFormValues';
import renderSuggestItemTitle from 'projects/trains/components/SearchForm/utilities/renderSuggestItemTitle';
import renderSuggestItemDescription from 'projects/trains/components/SearchForm/utilities/renderSuggestItemDescription';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getTrainsPinnedSegmentIdsAndDirectionForUrl} from 'projects/trains/components/SearchForm/utilities/getTrainsPinnedSegmentIdsAndDirectionForUrl';
import {isTransportCrossSearch} from 'utilities/searchForm/isCrossSearch';

import * as i18nBlock from 'i18n/components';
import * as i18nPriceCalendarBlock from 'i18n/components-PriceCalendar';
import * as i18nSearchForm from 'i18n/trains-SearchForm';

import CommonSearchForm, {
    IBaseSearchFormProps,
    IOnDateClickParams,
} from 'components/SearchForm/SearchForm';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import useSuggestInputFocusHandler from 'components/SearchForm/hooks/useSuggestInputFocusHandler';
import RoundTrip from 'components/RoundTripToggle/RoundTripToggle';
import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';
import {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';
import useCalendarConversionExpMetrics from 'components/DatePicker/hooks/useCalendarConversionExpMetrics';

import {useTrainsPriceCalendarProvider} from './hooks/useTrainsPriceCalendarProvider';

const FIELD_NAMES = [
    ESearchFormFieldName.FROM,
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
];

export const TRAINS_SPECIAL_WHEN_LIST = [
    WHEN_SPECIAL_VALUE.TODAY,
    WHEN_SPECIAL_VALUE.TOMORROW,
    WHEN_SPECIAL_VALUE.AFTER_TOMORROW,
];

interface ISearchFormProps extends IBaseSearchFormProps, IWithClassName {
    needToSetFromSuggestByGeoPosition?: boolean;
    query?: ParsedQuery;
    onSubmit?({searchLink}: {searchLink: string}): void;
    onChangeFromPoint?(value: ISuggestValue<ITrainsSuggest>): void;
    onChangeToPoint?(value: ISuggestValue<ITrainsSuggest>): void;
}

const SearchForm: React.FC<ISearchFormProps> = props => {
    const {
        needToSetFromSuggestByGeoPosition = false,
        query,
        fieldsRef,
        onSubmit,
        onChangeFromPoint,
        onChangeToPoint,
        ...restProps
    } = props;

    const {
        fromField,
        toField,
        startDateField,
        endDateField,
        isRoundTripField,
        searchSuggests,
        previousSearches,
        calendarPricesByType,
    } = useSelector(searchFormSelector);
    const context = useSelector(trainsContextSelector);
    const canUseNewCalendar = useNewCalendarEnabled();
    const {reachSearchInteraction, reachCalendarInteraction} =
        useCalendarConversionExpMetrics(ETrainsGoal);

    const dispatch = useDispatch();
    const deviceType = useDeviceType();

    const isInitialSuggestsRequestRef = useRef(true);
    const chosenPreviousSearch = useRef<ITrainsPreviousSearch | null>(null);

    const setFromPoint = useImmutableCallback(
        (point: ITrainsSearchFormPointField) => {
            dispatch(setTrainsSearchFormFromFieldAction(point));

            onChangeFromPoint?.(point);
        },
    );

    const setToPoint = useImmutableCallback(
        (point: ITrainsSearchFormPointField) => {
            dispatch(setTrainsSearchFormToFieldAction(point));

            onChangeToPoint?.(point);
        },
    );

    const setIsRoundTrip = useImmutableCallback((isRoundTrip: boolean) => {
        dispatch(setTrainsSearchFormIsRoundTripFieldAction(isRoundTrip));
    });

    const setStartDate = useImmutableCallback(
        (date: TTrainsSearchFormDateField) => {
            dispatch(setTrainsSearchFormStartDateFieldAction(date));
        },
    );

    const setEndDate = useImmutableCallback(
        (date: TTrainsSearchFormDateField) => {
            dispatch(setTrainsSearchFormEndDateFieldAction(date));

            if (date && !isRoundTripField) {
                setIsRoundTrip(true);
            }
        },
    );

    const setPreviousSearchFormValue = useImmutableCallback((data: any) => {
        dispatch(previousSearchFormSetFieldValue(data));
    });

    const requestSuggests = useImmutableCallback(
        (
            from: ITrainsSearchFormPointField,
            to: ITrainsSearchFormPointField,
        ) => {
            // Не фетчим саджесты во время загрузки cross-search
            if (
                from.source === ESuggestSource.CROSS_SEARCH &&
                to.source === ESuggestSource.CROSS_SEARCH &&
                !from.selectedValue &&
                !to.selectedValue
            ) {
                return;
            }

            dispatch(
                requestSearchSuggestsThunkAction({
                    fromField: from,
                    toField: to,
                    needToSetByGeoPointIfPossible:
                        needToSetFromSuggestByGeoPosition &&
                        isInitialSuggestsRequestRef.current,
                }),
            );

            isInitialSuggestsRequestRef.current = false;
        },
    );

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const handleReverseClick = useImmutableCallback(() => {
        reachGoal(ETrainsGoal.SEARCH_FORM_REVERSE_CLICK);
    });

    const handleCalendarShow = useImmutableCallback(
        (calendarType: ECalendarType) => {
            const oldStartCalendarOpened =
                !canUseNewCalendar && calendarType === ECalendarType.StartDate;
            const startCalendarOpened =
                oldStartCalendarOpened || canUseNewCalendar;

            if (!startCalendarOpened) {
                return;
            }

            reachGoal(ETrainsGoal.CALENDAR_OPEN);

            if (toField.selectedValue && fromField.selectedValue) {
                reachCalendarInteraction();
            }
        },
    );

    const handleDateClick = useImmutableCallback(
        (params?: IOnDateClickParams) => {
            if (!params) {
                return;
            }

            const {startDate} = params;

            if (!startDate) {
                return;
            }

            reachGoal(ETrainsGoal.CALENDAR_DATE_SELECTED);
        },
    );

    const handleSuggestInputFocus = useSuggestInputFocusHandler(
        fromField,
        toField,
        requestSuggestsWithThrottle,
    );

    const handleSelectSuggest = useImmutableCallback((item: ITrainsSuggest) => {
        if (!item.formValuesToSet) {
            return;
        }

        dispatch(
            fillSearchFormByPreviousSearchThunkAction(item.formValuesToSet),
        );

        if (deviceType.isMobile && !item.formValuesToSet.startDate) {
            setTimeout(() => {
                fieldsRef?.current?.focusFieldByName(
                    ESearchFormFieldName.START_DATE,
                );
            });
        }

        chosenPreviousSearch.current = item.formValuesToSet;

        reachGoal(ETrainsGoal.PREVIOUS_SEARCHES_SUGGEST_CLICK);
    });

    const handleChangeRoundTrip = useImmutableCallback(
        ({target: {value}}: {target: {value: 'oneWay' | 'roundTrip'}}) => {
            const isRoundTrip = value === 'roundTrip';

            setIsRoundTrip(isRoundTrip);

            if (!isRoundTrip) {
                setEndDate(null);
            }
        },
    );

    const footerBlockParams = useMemo((): IDatePickerFooterProps => {
        const buttonText =
            startDateField && !endDateField
                ? i18nSearchForm.oneWayTripButton()
                : i18nSearchForm.selectButton();
        const visible = Boolean(deviceType.isMobile && startDateField);
        const showButton = true;

        return {visible, showButton, buttonText};
    }, [startDateField, endDateField, deviceType.isMobile]);

    const footerBlockParamsByType =
        useMemo((): IDatePickerFooterPropsByCalendarType => {
            const hasStartDatePrices = !isEmpty(
                calendarPricesByType?.[ECalendarType.StartDate],
            );
            const hasEndDatePrices = !isEmpty(
                calendarPricesByType?.[ECalendarType.EndDate],
            );

            return {
                [ECalendarType.StartDate]: {
                    messageNode: i18nPriceCalendarBlock.forwardFooterLabel(),
                    visible: hasStartDatePrices,
                },
                [ECalendarType.EndDate]: {
                    messageNode: i18nPriceCalendarBlock.backwardFooterLabel(),
                    visible: hasEndDatePrices,
                },
            };
        }, [calendarPricesByType]);

    const handleFastDateClick = useImmutableCallback((value: string) => {
        switch (value) {
            case WHEN_SPECIAL_VALUE.TODAY:
                reachGoal(ETrainsGoal.CALENDAR_FASTDATE_TODAY);

                break;
            case WHEN_SPECIAL_VALUE.TOMORROW:
                reachGoal(ETrainsGoal.CALENDAR_FASTDATE_TOMORROW);

                break;

            case WHEN_SPECIAL_VALUE.AFTER_TOMORROW:
                reachGoal(ETrainsGoal.CALENDAR_FASTDATE_AFTERTOMORROW);

                break;
        }
    });

    const handleSubmit = useImmutableCallback(
        ({
            handleTrySubmit,
            formErrors,
            setHasSubmitFailed,
        }: {
            handleTrySubmit(): void;
            formErrors: ITrainsSearchFormErrors;
            setHasSubmitFailed(val: boolean): void;
        }) => {
            reachGoal(EIndexGoal.TRAINS_SEARCH_BUTTON);
            reachSearchInteraction();

            handleTrySubmit();

            if (isEmpty(formErrors)) {
                const fromSelectedValue = fromField.selectedValue;
                const toSelectedValue = toField.selectedValue;

                if (
                    typeof fromSelectedValue !== 'boolean' &&
                    typeof toSelectedValue !== 'boolean'
                ) {
                    const when =
                        startDateField === null
                            ? WHEN_SPECIAL_VALUE.TODAY
                            : startDateField;

                    const searchLink = trainsURLs.getTrainsSearchUrl(
                        {
                            context: {
                                from: fromSelectedValue.slug,
                                to: toSelectedValue.slug,
                                when: when || WHEN_SPECIAL_VALUE.TODAY,
                                returnWhen: endDateField,
                            },
                            lastSearchTimeMarker: String(Date.now()),
                            query: query as QueryInterface,
                            ...getTrainsPinnedSegmentIdsAndDirectionForUrl({
                                context,
                                toFieldSlug: toSelectedValue.slug,
                                fromFieldSlug: fromSelectedValue.slug,
                                isRoundTripField,
                                whenFieldValue: when,
                                returnWhenFieldValue: endDateField,
                            }),
                        },
                        {tld: TLD.RU},
                    );

                    history?.push(searchLink);

                    onSubmit?.({searchLink});

                    if (endDateField === null) {
                        dispatch(
                            setTrainsSearchFormIsRoundTripFieldAction(false),
                        );
                    }

                    if (startDateField === null) {
                        dispatch(
                            setTrainsSearchFormStartDateFieldAction(
                                WHEN_SPECIAL_VALUE.TODAY,
                            ),
                        );
                    }

                    const newPreviousSearch = {
                        from: fromSelectedValue,
                        to: toSelectedValue,
                        startDate: getWhenMoment(when).format(ROBOT),
                        endDate: endDateField
                            ? getWhenMoment(endDateField).format(ROBOT)
                            : null,
                    };

                    dispatch(addPreviousSearchThunkAction(newPreviousSearch));

                    if (
                        isEqual(chosenPreviousSearch.current, newPreviousSearch)
                    ) {
                        reachGoal(ETrainsGoal.PREVIOUS_SEARCHES_SUGGEST_SEARCH);

                        chosenPreviousSearch.current = null;
                    }

                    if (isTransportCrossSearch(fromField, toField)) {
                        reachGoal(ETrainsGoal.CROSS_SEARCH_SUBMIT);
                    }
                }
            } else {
                setHasSubmitFailed(true);
            }
        },
    );

    useEffect(() => {
        requestSuggestsWithThrottle(fromField, toField);
    }, [fromField, requestSuggestsWithThrottle, toField]);

    useEffect(() => {
        dispatch(restorePreviousSearchesThunkAction());
    }, [dispatch]);

    useTrainsPriceCalendarProvider({fromField, toField});

    const previousValues = getPreviousFormValues(previousSearches);

    const roundTripToggleBlock = useMemo(() => {
        return (
            <RoundTrip
                isRoundTrip={isRoundTripField}
                onChange={handleChangeRoundTrip}
            />
        );
    }, [handleChangeRoundTrip, isRoundTripField]);

    return (
        <CommonSearchForm
            searchFormName={RAILWAYS_SEARCH_FORM_NAME}
            fieldsNames={FIELD_NAMES}
            fromField={fromField}
            toField={toField}
            startDateField={startDateField}
            endDateField={endDateField}
            calendarPricesByType={calendarPricesByType}
            fromSearchSuggests={searchSuggests[ESearchFormFieldName.FROM].value}
            toSearchSuggests={searchSuggests[ESearchFormFieldName.TO].value}
            previousFormValues={previousValues}
            validateForm={validateForm}
            uniqueSuggestValueName="pointKey"
            getSuggestTitleAndDescription={getSuggestTitleAndDescription}
            renderSuggestItemTitle={renderSuggestItemTitle}
            renderSuggestItemDescription={renderSuggestItemDescription}
            fromPointPlaceholder={i18nBlock.railwaysSearchFormDotFromFieldPlaceholder()}
            toPointPlaceholder={i18nBlock.railwaysSearchFormDotToFieldPlaceholder()}
            startDatePlaceholder={i18nBlock.railwaysSearchFormDotStartDateTriggerPlaceholder()}
            endDatePlaceholder={i18nBlock.railwaysSearchFormDotEndDateTriggerPlaceholder()}
            specialWhenButtons={TRAINS_SPECIAL_WHEN_LIST}
            resetRangeAfterFirstInteraction
            datePickerHeaderBlock={roundTripToggleBlock}
            setFromPoint={setFromPoint}
            setToPoint={setToPoint}
            setStartDate={setStartDate}
            setEndDate={setEndDate}
            isRoundTrip={isRoundTripField || canUseNewCalendar}
            canAutoHideCalendar={
                canUseNewCalendar ? !deviceType.isMobile : undefined
            }
            datePickerFooterBlockParams={footerBlockParams}
            datePickerFooterBlockParamsByType={footerBlockParamsByType}
            hasEndDate
            setPreviousSearchFormValue={setPreviousSearchFormValue}
            fieldsRef={fieldsRef}
            onSelectSuggestItem={handleSelectSuggest}
            onReverseClick={handleReverseClick}
            onSuggestInputFocus={handleSuggestInputFocus}
            onFastDateClick={handleFastDateClick}
            onSubmit={handleSubmit}
            onShowCalendar={handleCalendarShow}
            onDateClick={handleDateClick}
            {...prepareQaAttributes('railways-search-form')}
            {...restProps}
        />
    );
};

export default React.memo(SearchForm);
