import {
    ITrainsSearchFormErrors,
    ITrainsSearchFormPointField,
    ITrainsSearchFormValues,
    TTrainsSearchFormDateField,
} from 'projects/trains/components/SearchForm/types';
import {ITrainsSuggest} from 'types/trains/common/ITrainsApiSuggest';
import {
    ESearchFormFieldName,
    ISearchFormSuggests,
} from 'components/SearchForm/types';

import {checkTrainsInvalidReturnWhen} from 'projects/trains/lib/date/checkTrainsInvalidReturnWhen';

import * as i18nBlock from 'i18n/components';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

const isEmptyDirectionValue = ({
    inputValue,
}: ITrainsSearchFormPointField): boolean => inputValue === '';

const isEmptySelectedValue = ({
    selectedValue,
}: ITrainsSearchFormPointField): boolean => selectedValue === false;

const isSelectedValue = (
    selectedValue: false | ITrainsSuggest,
): selectedValue is ITrainsSuggest => selectedValue !== false;

const isDirectionNotFound = ({
    inputValue,
    selectedValue,
}: ITrainsSearchFormPointField): boolean =>
    inputValue !== '' && selectedValue === false;

const isEqualDirections = (
    from: ITrainsSearchFormPointField,
    to: ITrainsSearchFormPointField,
): boolean => {
    const fromSelectedValue = from.selectedValue;
    const toSelectedValue = to.selectedValue;

    return (
        isSelectedValue(fromSelectedValue) &&
        isSelectedValue(toSelectedValue) &&
        fromSelectedValue.objId === toSelectedValue.objId
    );
};

const isInvalidStartDate = (
    startDate: TTrainsSearchFormDateField,
    endDate: TTrainsSearchFormDateField,
): boolean => {
    if (endDate) {
        return Boolean(!startDate);
    }

    return false;
};

/**
 * Валидация существует, потому что значение из cross-search подставляется еще до загрузки саджестов,
 * и перед поиском надо провалидировать, есть ли на самом деле такой саджест или нет
 */
const isUnknownCrossSearch = (
    field: ITrainsSearchFormPointField,
    suggests: ITrainsSuggest[] | null,
): boolean => {
    const {selectedValue, source} = field;

    if (!selectedValue || source !== ESuggestSource.CROSS_SEARCH || !suggests) {
        return false;
    }

    return suggests.every(suggest => {
        return suggest.pointKey !== selectedValue.pointKey;
    });
};

export default function validateForm(
    fieldsValues: ITrainsSearchFormValues,
    suggests: ISearchFormSuggests<ITrainsSuggest>,
): ITrainsSearchFormErrors {
    if (fieldsValues) {
        const {from, to, startDate, endDate} = fieldsValues;
        const errors: ITrainsSearchFormErrors = {};

        if (isEmptySelectedValue(from)) {
            errors.from = [
                i18nBlock.railwaysSearchFormDotValidateDotNeedSelectFromField(),
            ];
        }

        if (isEmptySelectedValue(to)) {
            errors.to = [
                i18nBlock.railwaysSearchFormDotValidateDotNeedSelectToField(),
            ];
        }

        if (isEmptyDirectionValue(from)) {
            errors.from = [
                i18nBlock.railwaysSearchFormDotValidateDotFromFieldRequired(),
            ];
        }

        if (isEmptyDirectionValue(to)) {
            errors.to = [
                i18nBlock.railwaysSearchFormDotValidateDotToFieldRequired(),
            ];
        }

        if (
            isDirectionNotFound(from) ||
            isUnknownCrossSearch(from, suggests[ESearchFormFieldName.FROM])
        ) {
            errors.from = [
                i18nBlock.railwaysSearchFormDotValidateDotUnknownFromDirection(),
            ];
        }

        if (
            isDirectionNotFound(to) ||
            isUnknownCrossSearch(to, suggests[ESearchFormFieldName.TO])
        ) {
            errors.to = [
                i18nBlock.railwaysSearchFormDotValidateDotUnknownToDirection(),
            ];
        }

        if (isEqualDirections(from, to)) {
            errors.to = [
                i18nBlock.railwaysSearchFormDotValidateDotEqualDirections(),
            ];
        }

        if (checkTrainsInvalidReturnWhen(startDate, endDate)) {
            errors.endDate = [
                i18nBlock.railwaysSearchFormDotValidateDotInvalidRangeDates(),
            ];
        }

        if (isInvalidStartDate(startDate, endDate)) {
            errors.startDate = [
                i18nBlock.railwaysSearchFormDotValidateDotStartDateRequired(),
            ];
        }

        return errors;
    }

    return {};
}
