import React from 'react';
import {Moment} from 'moment';
import moment from 'moment-timezone';

import {IWithClassName} from 'types/withClassName';
import ITrainsOrderInfoStation from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoStation';

import {HUMAN_SHORT, TIME} from 'utilities/dateUtils/formats';
import humanizeDuration from 'projects/trains/lib/date/duration/humanizeDuration';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-segment';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Station from 'projects/trains/components/TrainSegment/Station/Station';
import Separator from 'components/Separator/Separator';

import cx from './SegmentTimeAndStation.scss';

interface ISegmentTimeAndStationProps
    extends IWithClassName,
        IWithQaAttributes {
    departureDate: Moment;
    arrivalDate: Moment;

    departureStation: ITrainsOrderInfoStation;
    arrivalStation: ITrainsOrderInfoStation;
}

const SegmentTimeAndStation: React.FC<ISegmentTimeAndStationProps> = props => {
    const {
        className,
        departureDate,
        arrivalDate,
        departureStation,
        arrivalStation,
    } = props;

    const deviceType = useDeviceType();

    const departureIsNextDay = arrivalDate.isAfter(departureDate, 'day');

    const departureDateString = departureDate.format(HUMAN_SHORT);
    const arrivalDateString = arrivalDate.format(HUMAN_SHORT);

    const duration = arrivalDate.diff(departureDate, 'seconds');
    const durationMoment = moment.duration(duration, 'seconds');

    return (
        <Flex
            className={className}
            flexDirection={'column'}
            between={deviceType.isDesktop ? 3 : 0}
        >
            <Flex flexDirection={'column'}>
                <Flex
                    inline
                    justifyContent={'space-between'}
                    between={2}
                    alignItems={'baseline'}
                >
                    <Text
                        size={'s'}
                        color={'secondary'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departureDate',
                        })}
                    >
                        {departureDateString}
                    </Text>
                    <Text
                        size={'s'}
                        color={'secondary'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrivalDate',
                        })}
                    >
                        {departureIsNextDay
                            ? i18nBlock.nextDashDay({day: arrivalDateString})
                            : arrivalDateString}
                    </Text>
                </Flex>

                <Flex
                    inline
                    justifyContent={'space-between'}
                    between={2}
                    alignItems={'center'}
                >
                    <Text
                        size={'xl'}
                        weight={'bold'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departureTime',
                        })}
                    >
                        {departureDate.format(TIME)}
                    </Text>
                    <Separator className={cx('separator')} />
                    <Text>{humanizeDuration(durationMoment)}</Text>
                    <Separator className={cx('separator')} />
                    <Text
                        size={'xl'}
                        weight={'bold'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrivalTime',
                        })}
                    >
                        {arrivalDate.format(TIME)}
                    </Text>
                </Flex>
            </Flex>

            <Flex
                inline
                justifyContent={'space-between'}
                between={2}
                alignItems={'baseline'}
                nowrap
            >
                <Station
                    align={'left'}
                    station={
                        departureStation.popularTitle || departureStation.title
                    }
                />

                <Station
                    align={'right'}
                    station={
                        arrivalStation.popularTitle || arrivalStation.title
                    }
                />
            </Flex>
        </Flex>
    );
};

export default SegmentTimeAndStation;
