import React, {Component} from 'react';

import {
    ITrainFeature,
    TRAIN_FEATURES_TYPES,
} from 'projects/trains/lib/segments/features/types';
import {EBadgeTheme} from 'projects/trains/components/badges/Badge/types';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import ETicket from 'projects/trains/components/badges/ETicket/ETicket';
import DynamicPricing from 'projects/trains/components/badges/DynamicPricing/DynamicPricing';

import cx from './TrainSegmentInfo.scss';

interface IOwnProps extends IWithQaAttributes {
    trainFeatures: ITrainFeature[];
    deviceType: IDevice;
    className?: string;
    view?: string;
    tone?: string;
    size?: 'xs' | 'm' | 'xl';
    isBold?: boolean;
}

export default class TrainSegmentInfo extends Component<IOwnProps> {
    static defaultProps = {
        size: 'xs',
    };

    render(): React.ReactNode {
        const {trainFeatures, deviceType, className, view, tone, size, isBold} =
            this.props;

        if (!trainFeatures || !trainFeatures.length) {
            return null;
        }

        return (
            <div className={cx('root', className)}>
                {trainFeatures.map(({type, props = {}, content}) => {
                    const itemClassName = cx(
                        'item',
                        `item_${type}`,
                        view && `item_view_${view}`,
                        tone && `item_tone_${tone}`,
                        size && `item_size_${size}`,
                        {
                            item_mobile: deviceType.isMobile,
                            item_isBold: isBold,
                        },
                    );

                    switch (type) {
                        case TRAIN_FEATURES_TYPES.ETICKET:
                            return (
                                <ETicket
                                    className={cx(itemClassName, 'badge')}
                                    key={type}
                                    theme={EBadgeTheme.LIGHT}
                                    {...props}
                                    {...prepareQaAttributes(this.props)}
                                />
                            );
                        case TRAIN_FEATURES_TYPES.DYNAMIC_PRICING:
                            return (
                                <DynamicPricing
                                    className={cx(itemClassName, 'badge')}
                                    key={type}
                                    theme={EBadgeTheme.LIGHT}
                                    {...props}
                                />
                            );
                        case TRAIN_FEATURES_TYPES.COMPANY:
                            return (
                                <span
                                    className={cx(itemClassName, {
                                        item_desktop: deviceType.isDesktop,
                                    })}
                                    key={type}
                                    {...prepareQaAttributes({
                                        parent: this.props,
                                        current: 'company',
                                    })}
                                    {...props}
                                >
                                    {content}
                                </span>
                            );
                        case TRAIN_FEATURES_TYPES.MODEL:
                        case TRAIN_FEATURES_TYPES.SUBTYPE:
                        case TRAIN_FEATURES_TYPES.TWO_STOREY:
                        case TRAIN_FEATURES_TYPES.TROUGH_TRAIN:
                        case TRAIN_FEATURES_TYPES.TRAIN_CATEGORY:
                        case TRAIN_FEATURES_TYPES.DELUXE_TRAIN:
                            return (
                                <span
                                    className={itemClassName}
                                    key={type}
                                    {...(type ===
                                    TRAIN_FEATURES_TYPES.DELUXE_TRAIN
                                        ? prepareQaAttributes({
                                              parent: this.props,
                                              current: 'firm',
                                          })
                                        : {})}
                                    {...props}
                                >
                                    {content}
                                </span>
                            );

                        default:
                            return content;
                    }
                })}
            </div>
        );
    }
}
