import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './Station.scss';

interface IStationProps extends IWithClassName, IWithQaAttributes {
    station: string;
    align?: 'left' | 'right';
    city?: string;
    cityIsLoading?: boolean;
    cityIsVisible?: boolean;
    isVisible?: boolean;
}

const Station: React.FC<IStationProps> = props => {
    const {
        station,
        city,
        className,
        align = 'left',
        cityIsLoading = false,
        isVisible = true,
        cityIsVisible = false,
    } = props;

    const deviceType = useDeviceType();
    const cityIsEqualStation = city === station;

    return (
        <Flex
            className={cx('root', align, className)}
            flexDirection="column"
            alignItems={align === 'left' ? 'flex-start' : 'flex-end'}
            {...prepareQaAttributes(props)}
        >
            {isVisible && (
                <>
                    {cityIsVisible &&
                        !cityIsEqualStation &&
                        (cityIsLoading ? (
                            <TextSkeleton
                                className={cx('citySkeleton')}
                                size="m"
                                withAnimation
                            />
                        ) : (
                            city && (
                                <Text
                                    weight={
                                        deviceType.isMobile ? 'normal' : 'bold'
                                    }
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'city',
                                    })}
                                >
                                    {city}
                                </Text>
                            )
                        ))}
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'station',
                        })}
                    >
                        {station}
                    </span>
                </>
            )}
        </Flex>
    );
};

export default Station;
