import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    ITrainFeature,
    TRAIN_FEATURES_TYPES,
} from 'projects/trains/lib/segments/features/types';

import filterTrainFeatures from 'projects/trains/lib/segments/features/filterTrainFeatures';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';

import * as i18nBlock from 'i18n/trains-transport-types';

import Text from 'components/Text/Text';
import TrainSegmentInfo from 'projects/trains/components/TrainSegment/SegmentInfo/TrainSegmentInfo';
import DotSeparator from 'components/DotSeparator/DotSeparator';

import cx from './TrainsSegmentTitleMobile.scss';

export interface ITrainSegmentTitleMobileProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    segment:
        | {}
        | {
              number?: string;
              title?: string;
              isMetaSegment?: boolean;
          };
    features: ITrainFeature[];
    canHideCompany?: boolean;
}

const TrainSegmentTitleMobile: React.FC<ITrainSegmentTitleMobileProps> =
    props => {
        const {className, segment, features, deviceType, canHideCompany} =
            props;
        const companyTrainFeatures = features.filter(
            feature => feature.type === TRAIN_FEATURES_TYPES.COMPANY,
        );
        const visibleFeatures = filterTrainFeatures(features, segment);

        return (
            <div className={cx('root', className)}>
                <div className={cx('information')}>
                    <Text>
                        {companyTrainFeatures.length > 0 && !canHideCompany && (
                            <TrainSegmentInfo
                                className={cx('company')}
                                trainFeatures={companyTrainFeatures}
                                deviceType={deviceType}
                                isBold
                                size="m"
                                {...prepareQaAttributes(props)}
                            />
                        )}

                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'numberAndDirection',
                            })}
                        >
                            {`${capitalizeFirstLetter(
                                i18nBlock.typeDashTrain(),
                            )} ${'number' in segment ? segment.number : ''}, ${
                                'title' in segment ? segment.title : ''
                            }`}
                        </span>

                        {visibleFeatures.length > 0 && (
                            <>
                                <DotSeparator />
                                <TrainSegmentInfo
                                    className={cx('transport')}
                                    trainFeatures={visibleFeatures}
                                    deviceType={deviceType}
                                    size="m"
                                    {...prepareQaAttributes(props)}
                                />
                            </>
                        )}
                    </Text>
                </div>
            </div>
        );
    };

export default TrainSegmentTitleMobile;
