import React, {useEffect} from 'react';
import {Route, Switch} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {URLs} from 'constants/urls';

import {ELoadableReducer} from 'types/common/ELoadableReducer';

import resetSearchAndContext from 'reducers/trains/actions/resetSearchAndContext';
import popularRoutesReducer from 'reducers/trains/popularRoutes/reducer';
import orderReducer from 'reducers/trains/order/reducer';
import viewReducer from 'reducers/trains/view/reducer';
import searchReducer from 'reducers/trains/search/reducer';
import contextReducer from 'reducers/trains/context/reducer';
import userActionsReducer from 'reducers/trains/logging/reducer';
import {trainsPartner as partnerReducer} from 'reducers/trains/common/reducer';
import happyPageReducer from 'reducers/happyPage/reducer';
import genericSearchReducer from 'reducers/trains/genericSearch/reducer';
import searchSuggestsReducer from 'reducers/trains/searchSuggests/reducer';
import previousSearchesReducer from 'reducers/trains/previousSearches/reducer';
import priceCalendarReducer from 'reducers/trains/priceCalendar/reducer';

import trainsSaga from 'sagas/trains/rootSaga';
import requestHappyPageSaga from 'sagas/happyPage/orderAndCrossSales/requestHappyPageSaga';

import {loadable} from 'utilities/pageLoadable';
import isTrainsWhenSearch from 'projects/trains/lib/date/isTrainsWhenSearch';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AppRoute from 'components/AppRoute/AppRoute';
import {TrainsPopularRoutesMeta} from 'projects/trains/components/meta/TrainsPopularRoutesMeta';
import TrainsRedirectOldSearch from 'projects/trains/components/TrainsApp/RedirectOldSearch';
import TrainsPartnerRedirectPage from 'projects/trains/components/TrainsPartnerRedirectPage';
import TrainsRedirectOldSearchWithWhenParam from 'projects/trains/components/TrainsApp/TrainsRedirectOldSearchWithWhenParam';
import HangingYandexMessenger from 'projects/trains/components/HangingYandexMessenger/HangingYandexMessenger';

const TrainsSearchPage = loadable(
    () =>
        import(
            'projects/trains/pages/TrainsSearchPage/TrainsSearchPageContainer'
        ),
);

const DirectionSearchPage = loadable(
    () =>
        import('projects/trains/pages/DirectionSearchPage/DirectionSearchPage'),
);

const TrainsOrderApp = loadable(
    () => import('projects/trains/components/TrainsOrderApp/TrainsOrderApp'),
);

const TrainsPopularRoutesPage = loadable(
    () => import('projects/trains/components/TrainsPopularRoutesPage'),
);

const HappyPageLoadable = loadable(
    () => import('projects/trains/pages/HappyPage/HappyPage'),
);

const FinishPaymentPage = loadable(
    () => import('components/Layouts/PaymentFinish/PaymentFinish'),
);

const AuthorizeTicketPage = loadable(
    () => import('containers/AuthTicket/AuthTicket'),
);

const FirmTrainPage = loadable(
    () => import('projects/trains/pages/FirmTrainPage/FirmTrainPage'),
);

const NotFoundPage = loadable(
    () => import('components/NotFound/NotFoundPage/NotFoundPage'),
);

const SUPPORT_CHAT_PAGES = [
    URLs.trainsSearch,
    URLs.trainsOrderPlaces,
    URLs.trainsOrderPassengers,
    URLs.trainsOrderConfirm,
    URLs.trainsOrderHappyPage,
    URLs.trainsPopularRoutes,
    URLs.trainsFirmTrain,
];

const TrainsApp: React.FC = () => {
    const dispatch = useDispatch();

    useEffect(
        () => (): void => {
            dispatch(resetSearchAndContext());
        },
        [dispatch],
    );

    return (
        <>
            <Switch>
                <Route
                    exact
                    path={URLs.trainsOldSearch}
                    component={TrainsRedirectOldSearch}
                />

                <Route
                    exact
                    path={URLs.trainsOldSearchWithWhenParam}
                    component={TrainsRedirectOldSearchWithWhenParam}
                />

                <Route
                    exact
                    path={URLs.trainsSearch}
                    render={({location}): React.ReactNode => {
                        const {when} = getQueryByLocation(location, false);

                        if (typeof when !== 'string') {
                            return <DirectionSearchPage />;
                        } else if (!isTrainsWhenSearch(when)) {
                            return <NotFoundPage />;
                        }

                        return <TrainsSearchPage />;
                    }}
                />

                <Route path={URLs.trainsOrderHappyPage}>
                    <HappyPageLoadable />
                </Route>

                <Route path={URLs.trainsOrderFinishPayment}>
                    <FinishPaymentPage />
                </Route>

                <Route path={URLs.trainsOrder} component={TrainsOrderApp} />

                <Route
                    path={URLs.trainsPartnerRedirect}
                    component={TrainsPartnerRedirectPage}
                />

                <AppRoute
                    path={URLs.trainsPopularRoutes}
                    component={TrainsPopularRoutesPage}
                    meta={TrainsPopularRoutesMeta}
                />

                <Route
                    path={URLs.trainsAuthorizeTicket}
                    component={AuthorizeTicketPage}
                />

                <Route path={URLs.trainsFirmTrain}>
                    <FirmTrainPage />
                </Route>

                <Route component={NotFoundPage} />
            </Switch>

            <Route exact path={SUPPORT_CHAT_PAGES}>
                <HangingYandexMessenger />
            </Route>
        </>
    );
};

export default withReducers([
    [ELoadableReducer.TRAINS_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.TRAINS_POPULAR_ROUTES, popularRoutesReducer],
    [ELoadableReducer.TRAINS_ORDER, orderReducer],
    [ELoadableReducer.TRAINS_VIEW, viewReducer],
    [ELoadableReducer.TRAINS_SEARCH, searchReducer],
    [ELoadableReducer.TRAINS_CONTEXT, contextReducer],
    [ELoadableReducer.TRAINS_USER_ACTIONS, userActionsReducer],
    [ELoadableReducer.TRAINS_PARTNER, partnerReducer],
    [ELoadableReducer.HAPPY_PAGE, happyPageReducer],
    [ELoadableReducer.TRAINS_GENERIC_SEARCH, genericSearchReducer],
    [ELoadableReducer.TRAINS_PREVIOUS_SEARCHES, previousSearchesReducer],
    [ELoadableReducer.TRAINS_PRICE_CALENDAR, priceCalendarReducer],
])(withSaga([trainsSaga, requestHappyPageSaga])(TrainsApp));
