import React, {Component, Fragment} from 'react';
import {connect} from 'react-redux';
import {
    Redirect,
    Route,
    RouteComponentProps,
    Switch,
    withRouter,
} from 'react-router-dom';
import {parse} from 'query-string';
import isEqual from 'lodash/isEqual';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {URLs} from 'constants/urls';

import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {IWithDeviceType} from 'types/withDeviceType';
import {EPubSubEvent} from 'types/EPubSubEvent';

import trainsInitOrderApp from 'reducers/trains/order/thunk/trainsInitOrderApp';
import changeOrderStep, {
    TOrderStepDescription,
} from 'reducers/trains/order/thunk/changeOrderStep';
import {StoreInterface} from 'reducers/storeTypes';

import shouldUseOwnOrderPageSelector from 'selectors/trains/shouldUseOwnOrderPageSelector';
import deviceTypeSelector from 'selectors/common/deviceTypeSelector';
import currentOrderStepDescriptionSelector from 'selectors/trains/order/currentOrderStepDescriptionSelector';

import {reachGoalOnce} from 'utilities/metrika';
import renderOrderStep from 'projects/trains/components/TrainsOrderApp/helpers/renderOrderStep';
import {getOrderError} from 'projects/trains/lib/errors/orderErrors';
import {trainsURLs} from 'projects/trains/lib/urls';
import {getTrainsOrderParamsByQuery} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {getOrderStepDescriptionByPath} from 'projects/trains/lib/urls/getOrderStepUrl';
import {publish} from 'utilities/pubSub';

import ErrorBoundary from 'components/ErrorBoundary/ErrorBoundary';
import NotFoundFragment from 'components/NotFound/NotFoundFragment/NotFoundFragment';
import TrainsOrderPage from 'projects/trains/components/TrainsOrderPage/TrainsOrderPage';
import OrderError from 'projects/trains/components/TrainsOrderPage/OrderError';
import TrainsOrderLayout from 'projects/trains/components/layouts/TrainsOrderLayout';
import TrainsOrderErrorPage from 'projects/trains/components/TrainsOrderPage/ErrorPage/ErrorPage';
import BookError from 'projects/trains/components/TrainsOrderApp/components/BookError/BookError';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import {fetchOrderData} from 'server/redux/trains/order';

interface ITrainsOrderAppStateProps extends IWithDeviceType {
    shouldUseOwnOrderPage: boolean;
    orderStepDescription: TOrderStepDescription;
}

const mapStateToProps = (state: StoreInterface): ITrainsOrderAppStateProps => ({
    orderStepDescription: currentOrderStepDescriptionSelector(state),
    shouldUseOwnOrderPage: shouldUseOwnOrderPageSelector(state),
    deviceType: deviceTypeSelector(state),
});

const mapDispatchToProps = {
    changeOrderStep,
    initOrderApp: trainsInitOrderApp,
};

type TTrainsOrderAppProps = ITrainsOrderAppStateProps &
    typeof mapDispatchToProps &
    RouteComponentProps;

class TrainsOrderApp extends Component<TTrainsOrderAppProps> {
    componentDidMount(): void {
        const {location, initOrderApp} = this.props;
        const orderStepDescription = getOrderStepDescriptionByPath(
            location.pathname,
        );

        initOrderApp(
            orderStepDescription,
            getTrainsOrderParamsByQuery(getQueryByLocation(location)),
        );
    }

    componentDidUpdate(prevProps: TTrainsOrderAppProps): void {
        const {pathname: prevPathname} = prevProps.location;
        const {pathname} = this.props.location;

        if (pathname === prevPathname) {
            return;
        }

        this.handleOrderStepChanging();
    }

    componentWillUnmount(): void {
        publish(EPubSubEvent.TRAINS_LEAVE_BOOK_PAGES);
    }

    private handleOrderStepChanging(): void {
        const {
            location: {pathname},
            orderStepDescription,
        } = this.props;
        const pathStepDescription = getOrderStepDescriptionByPath(pathname);

        if (isEqual(orderStepDescription, pathStepDescription)) {
            return;
        }

        this.props.changeOrderStep(pathStepDescription);
    }

    private handleOrderPageRenderError = (): void => {
        reachGoalOnce(ETrainsGoal.ORDER_RENDER_ERROR);
    };

    private renderOrderPageRenderError = (): React.ReactNode => {
        const {deviceType} = this.props;

        return (
            <TrainsOrderLayout deviceType={deviceType}>
                {{
                    allWidth: (
                        <TrainsOrderErrorPage errorData={getOrderError(null)} />
                    ),
                }}
            </TrainsOrderLayout>
        );
    };

    private renderOrderPage = (): React.ReactNode => {
        return (
            <ErrorBoundary
                renderError={this.renderOrderPageRenderError}
                handleError={this.handleOrderPageRenderError}
            >
                <TrainsOrderPage />
            </ErrorBoundary>
        );
    };

    render(): React.ReactNode {
        const {orderStepDescription, shouldUseOwnOrderPage, location} =
            this.props;

        if (!shouldUseOwnOrderPage) {
            return (
                <Redirect
                    to={trainsURLs.getTrainsPartnerRedirectUrl(
                        parse(location.search),
                    )}
                />
            );
        }

        return (
            <Fragment>
                <Switch>
                    <Route
                        path={URLs.trainsOrderPayment}
                        render={renderOrderStep(
                            orderStepDescription.step,
                            ORDER_STEP.PAYMENT,
                        )}
                    />

                    <Route
                        path={URLs.trainsOrder}
                        render={this.renderOrderPage}
                    />

                    <Route component={NotFoundFragment} />
                </Switch>

                <OrderError />
                <BookError />
            </Fragment>
        );
    }
}

export default withRouter(
    serverFetchDataDispatcher([fetchOrderData], {checkNested: true})(
        connect(mapStateToProps, mapDispatchToProps)(TrainsOrderApp),
    ),
);
