import {FC, memo, useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {ETrainApiOrderErrorCode} from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/TTrainOrderError/TTrainApiOrderError';
import {ETrainOrderErrorCode} from 'projects/trains/components/TrainsOrderApp/components/BookError/types';

import {StoreInterface} from 'reducers/storeTypes';
import choosePlacesAgain from 'reducers/trains/order/thunk/choosePlacesAgain';
import {setBookError} from 'reducers/trains/order/actions/view';
import changeOrderStep from 'reducers/trains/order/thunk/changeOrderStep';

import {trainsURLs} from 'projects/trains/lib/urls';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import getAction from 'projects/trains/components/TrainsOrderApp/components/BookError/utilities/getAction';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18n from 'i18n/trains-BookError';

import ErrorModal, {IErrorModalProps} from 'components/ErrorModal/ErrorModal';
import Link from 'components/Link/Link';
import SupportPhone from 'components/SupportPhone/SupportPhone';

interface IBookErrorProps {}

const BookError: FC<IBookErrorProps> = () => {
    const dispatch = useDispatch();
    const bookError = useSelector(
        (state: StoreInterface) => state.trains.order.bookError,
    );

    const changePlaces = useCallback(() => {
        dispatch(setBookError(null));
        dispatch(choosePlacesAgain());
    }, [dispatch]);

    const changePassengers = useCallback(() => {
        dispatch(setBookError(null));
        dispatch(changeOrderStep({step: ORDER_STEP.PASSENGERS}));
    }, [dispatch]);

    const changePlacesAction = useMemo(
        () => getAction(i18n.changePlacesAction(), changePlaces),
        [changePlaces],
    );

    const changePassengersAction = useMemo(
        () => getAction(i18n.changePassengersAction(), changePassengers),
        [changePassengers],
    );

    const errorInfo = useMemo((): Pick<
        IErrorModalProps,
        'title' | 'text' | 'primaryAction' | 'secondaryAction'
    > | null => {
        if (!bookError) {
            return null;
        }

        if (bookError.code === ETrainOrderErrorCode.COMMON) {
            return {
                title: i18n.commonTitle(),
                primaryAction: changePlacesAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.NO_PLACES) {
            return {
                title: i18n.commonTitle(),
                text: i18n.noPlacesError(),
                primaryAction: changePlacesAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.INVALID_BONUS_CARD) {
            return {
                title: i18n.commonTitle(),
                text: i18n.invalidBonusCardError(),
                primaryAction: changePassengersAction,
            };
        }

        if (
            bookError.code ===
            ETrainApiOrderErrorCode.NAME_REQUIRED_LATIN_LETTERS
        ) {
            return {
                title: i18n.commonTitle(),
                text: i18n.nameRequiredLatinLettersError(),
                primaryAction: changePassengersAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.TARIFF_ERROR) {
            return {
                title: i18n.commonTitle(),
                text: i18n.tariffError(),
                primaryAction: changePassengersAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.TRY_LATER) {
            return {
                title: i18n.commonTitle(),
                text: i18n.tryLaterError(),
                primaryAction: changePassengersAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.TOO_LATE) {
            return {
                title: i18n.commonTitle(),
                text: i18n.tooLateError(),
                primaryAction: changePassengersAction,
            };
        }

        if (
            bookError.code === ETrainApiOrderErrorCode.INVALID_DOCUMENT_NUMBER
        ) {
            return {
                title: i18n.commonTitle(),
                text: i18n.invalidDocumentNumberError(),
                primaryAction: changePassengersAction,
            };
        }

        if (bookError.code === ETrainApiOrderErrorCode.TOO_LATE_FOR_ORDER) {
            return {
                title: i18n.commonTitle(),
                text: i18n.tooLateForOrderError(),
                primaryAction: changePlacesAction,
            };
        }

        if (
            bookError.code ===
            ETrainApiOrderErrorCode.CITIZENSHIP_NOT_MATCH_DOCUMENT_TYPE
        ) {
            return {
                title: i18n.commonTitle(),
                text: i18n.citizenshipNotMatchDocumentTypeError(),
                primaryAction: changePassengersAction,
            };
        }

        return {
            title: i18n.commonTitle(),
            text: insertJSXIntoKey(i18n.unknownPartnerError)({
                phone: <SupportPhone />,
                feedbackLink: (
                    <Link
                        theme="normal"
                        target="_blank"
                        url={trainsURLs.getOrderFeedbackUrl()}
                    >
                        {i18n.unknownPartnerErrorLink()}
                    </Link>
                ),
            }),
            primaryAction: changePlacesAction,
        };
    }, [bookError, changePassengersAction, changePlacesAction]);

    if (!errorInfo) {
        return null;
    }

    return (
        <ErrorModal
            isVisible
            {...errorInfo}
            {...prepareQaAttributes('orderError')}
        />
    );
};

export default memo(BookError);
