import React, {Component, MouseEvent} from 'react';
import {connect} from 'react-redux';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {
    mapOrderStepToInvalidSubmitGoal,
    mapOrderStepToValidSubmitGoal,
    mapRestrictionTypeToGoal,
} from 'projects/trains/constants/metrika';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import changeOrderStepAction from 'reducers/trains/order/thunk/changeOrderStep';
import {DispatchProps} from 'reducers/trains/customDispatch';

import orderButtonSelector from 'projects/trains/components/TrainsOrderButton/selectors/orderButtonSelector';

import {reachGoalOnce} from 'utilities/metrika';
import {orderMessages} from 'projects/trains/lib/order/messages';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getActionButtonText from './utilities/getActionButtonText';

import Button, {IButtonProps} from 'components/Button/Button';
import TrainsOrderRestrictionsMessage from 'projects/trains/components/Messages/TrainsOrderRestrictionsMessage';
import HintDesktop from 'projects/trains/components/Hint/components/HintDesktop/HintDesktop';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import sendOnValidClickInPlacesMetrika from 'projects/trains/components/TrainsOrderButton/thunk/sendOnValidClickInPlacesMetrika';

import cx from './TrainsOrderButton.scss';

const CLICK_TIMEOUT = 1000;
const POPUP_POSITIONS = [
    EPopupDirection.TOP,
    EPopupDirection.BOTTOM,
    EPopupDirection.TOP_RIGHT,
    EPopupDirection.BOTTOM_RIGHT,
];

interface IOwnProps extends IWithClassName, IWithQaAttributes {
    hintClassName?: string;
    hintText?: string;
    buttonClassName?: string;
    buttonProps?: IButtonProps;
    onClick?(): void;
    innerRef?: React.Ref<HTMLButtonElement>;
}

type TPropsType = IOwnProps &
    ReturnType<typeof orderButtonSelector> &
    DispatchProps;

type TStateType = {
    messageIsVisible: boolean;
};

class TrainsOrderButton extends Component<TPropsType, TStateType> {
    state: TStateType = {
        messageIsVisible: false,
    };

    protected _rootRef: React.RefObject<HTMLDivElement> = React.createRef();
    protected _buttonRef: React.RefObject<HTMLDivElement> = React.createRef();

    /**
     * Таймаут, чтобы нельзя было нечаянно сделать двойной клик.
     *
     * @see https://st.yandex-team.ru/TRAVELFRONT-1023
     *
     * @type {number|undefined}
     */
    protected clickTimeout: number | undefined;
    protected attentionTimeout: number | undefined;

    componentDidMount(): void {
        document.addEventListener('touchstart', this.hideMessage);
    }

    // eslint-disable-next-line camelcase
    UNSAFE_componentWillReceiveProps(nextProps: TPropsType): void {
        if (!nextProps.disabled) {
            this.hideMessage();
        }
    }

    componentWillUnmount(): void {
        document.removeEventListener('touchstart', this.hideMessage);

        clearTimeout(this.attentionTimeout);
        clearTimeout(this.clickTimeout);
    }

    protected onClick = (e: MouseEvent<HTMLButtonElement>): void => {
        if (this.clickTimeout) {
            e.preventDefault();
            e.stopPropagation();

            return;
        }

        this.clickTimeout = window.setTimeout(() => {
            this.clickTimeout = undefined;
        }, CLICK_TIMEOUT);

        if (this.props.onClick) {
            this.props.onClick();
        }

        // На событие blur полей ввода данных пассажиров происходит выставление данных.
        // В ie blur произойдет после клика по кнопке перехода на следующий шаг покупки,
        // но в handleClick нам нужны актуальные данные, поэтому запускаем асинхронно.
        // Как вариант, можно сделать выставление на debounce onChange.
        window.setTimeout(this.handleClick);
    };

    private handleClick = (): void => {
        const {disabled, solutions, restrictions} = this.props;

        if (disabled) {
            if (solutions.length === restrictions.length) {
                solutions.forEach(solution => {
                    this.props.dispatch(solution());
                });
                this.onValidClick();
            } else {
                this.onInvalidClick();
            }
        } else {
            this.onValidClick();
        }
    };

    private async onValidClick(): Promise<void> {
        const {dispatch, orderStep, nextOrderStep} = this.props;

        reachGoalOnce(mapOrderStepToValidSubmitGoal[orderStep]);

        switch (orderStep) {
            case ORDER_STEP.PLACES: {
                dispatch(sendOnValidClickInPlacesMetrika);

                break;
            }

            case ORDER_STEP.PASSENGERS: {
                return;
            }
        }

        dispatch(changeOrderStepAction(nextOrderStep));
    }

    private onInvalidClick(): void {
        const {orderStep} = this.props;
        const goal = mapOrderStepToInvalidSubmitGoal[orderStep];

        if (goal) {
            reachGoalOnce(goal);
        }

        this.showMessage();
    }

    private onClickOutside = (): void => {
        this.hideMessage();
    };

    private showMessage(): void {
        const {restrictions} = this.props;

        this.setState({
            messageIsVisible: true,
        });

        if (orderMessages.has(restrictions[0])) {
            const goal = mapRestrictionTypeToGoal[restrictions[0].type];

            if (goal) {
                reachGoalOnce(goal);
            }
        }
    }

    private hideMessage = (): void => {
        this.setState({messageIsVisible: false});
    };

    private renderMessage(): React.ReactNode {
        const {disabled, restrictions} = this.props;
        const {messageIsVisible} = this.state;

        if (!orderMessages.has(restrictions[0])) {
            return null;
        }

        return (
            <MessageBoxPopup
                scopeRef={this._rootRef}
                isVisible={disabled && messageIsVisible}
                anchorRef={this._buttonRef}
                direction={POPUP_POSITIONS}
                onClose={this.onClickOutside}
            >
                <TrainsOrderRestrictionsMessage restrictions={restrictions} />
            </MessageBoxPopup>
        );
    }

    render(): React.ReactNode {
        const {messageIsVisible} = this.state;
        const {
            className,
            hintText,
            hintClassName,
            buttonClassName,
            buttonProps,
            orderStep,
            nextOrderStep,
            innerRef,
        } = this.props;
        const isSubmitButton = orderStep === ORDER_STEP.PASSENGERS;

        const button = (
            <div ref={this._buttonRef}>
                <Button
                    innerRef={innerRef}
                    className={cx(buttonClassName)}
                    {...prepareQaAttributes(this.props)}
                    size="xl"
                    theme="primary"
                    width="max"
                    type={isSubmitButton ? 'submit' : 'button'}
                    {...buttonProps}
                    onClick={this.onClick}
                >
                    {getActionButtonText(nextOrderStep)}
                </Button>
            </div>
        );

        return (
            <div className={cx('root', className)} ref={this._rootRef}>
                {this.renderMessage()}

                {messageIsVisible || !hintText ? (
                    button
                ) : (
                    <HintDesktop className={hintClassName} text={hintText}>
                        {button}
                    </HintDesktop>
                )}
            </div>
        );
    }
}

export default connect(orderButtonSelector)(TrainsOrderButton);
