import React, {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ICoachesTab} from 'projects/trains/components/TrainsOrderPage/Coaches/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {changeCoachAndResetRequirements} from 'reducers/trains/actions/changeCoachAndResetRequirements';

import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';
import currentAdditionalSchemeInfoSelector from 'selectors/trains/order/currentAdditionalSchemeInfoSelector';

import {reachGoalOnce} from 'utilities/metrika';
import getCoachClassKey from 'projects/trains/lib/order/getCoachClassKey';
import checkSchemeHasGroupedPlaces from 'projects/trains/lib/order/coaches/checkSchemeHasGroupedPlaces';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import CoachHeader from './components/CoachHeader/CoachHeader';
import CoachFooter from './components/CoachFooter/CoachFooter';
import CoachAttentions from './components/CoachAttentions/CoachAttentions';
import CoachAutoSeatButton from './components/CoachAutoSeatButton/CoachAutoSeatButton';
import CoachAdditionalInfo from './components/CoachAdditionalInfo/CoachAdditionalInfo';
import CoachTransportSchema from './components/CoachTransportSchema/CoachTransportSchema';
import OrderSummaryCompact from 'projects/trains/components/OrderSummaryCompact/OrderSummaryCompact';
import CoachNonRefundableDiscount from './components/CoachNonRefundableDiscount/CoachNonRefundableDiscount';

import {useScrollToPassengersStepButton} from './hooks/useScrollToPassengersStepButton';
import {useCoachSchemeLogAndCatchCallback} from './hooks/useCoachSchemeLogAndCatchCallback';

import cx from './Coach.scss';

interface ICoachProps extends ICoachesTab, IWithClassName, IWithQaAttributes {
    isLast: boolean;
}

const Coach: React.FC<ICoachProps> = props => {
    const {
        className,
        coach,
        scale,
        active,
        coachSchemaRenderingDelay,
        withoutCoachNumberTitle,
        schema,
        schemaPlaceMaskPrefix,
        gender,
        segment,
        passengers,
        showNFCoach,
        orderPlaces,
        isDifferentCoachesArrival,
        isLast,
    } = props;

    const dispatch = useDispatch();
    const {isMobile, isDesktop} = useDeviceType();
    const passengersStepButtonRef = useScrollToPassengersStepButton({
        isMobile,
        passengers,
        orderPlaces,
    });
    const handleCoachChange = useCallback(() => {
        if (active) {
            return;
        }

        dispatch(changeCoachAndResetRequirements(coach));

        reachGoalOnce(ETrainsGoal.COACH_NUMBER_WAS_CHANGED);
    }, [dispatch, active, coach]);
    const additionalSchemeInfo = useSelector(
        currentAdditionalSchemeInfoSelector,
    );
    const orderStep = useSelector(orderStepSelector);
    const schemeHasGroupedPlaces = checkSchemeHasGroupedPlaces(
        coach,
        additionalSchemeInfo,
    );
    const summaryIsVisible = isMobile && active && orderPlaces.length !== 0;
    const coachWithoutSchema = !(coach.schemaId && schema);

    const handleSchemeCatchError = useCoachSchemeLogAndCatchCallback({
        coach,
        segment,
        coachWithoutSchema,
    });

    return (
        <li
            className={cx('root', className, {
                root_isMobile: isMobile,
                root_active: true,
            })}
            {...prepareQaAttributes(props)}
        >
            <div
                role="button"
                className={cx('button')}
                onClick={handleCoachChange}
            >
                <CoachHeader
                    className={cx('header')}
                    coach={coach}
                    showNFCoach={showNFCoach}
                    withoutCoachNumberTitle={withoutCoachNumberTitle}
                    showCoachArrival={isDifferentCoachesArrival}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'coachHeader',
                    })}
                />
                <CoachAdditionalInfo
                    coach={coach}
                    previewType={isDesktop ? 'full' : 'short'}
                />
                <Flex
                    className={cx('attentions')}
                    justifyContent={isDesktop ? 'space-between' : 'flex-start'}
                    flexDirection={isDesktop ? 'row' : 'column'}
                    alignItems={isDesktop ? 'center' : 'flex-start'}
                    inline={isDesktop}
                    above={4}
                    between={isDesktop ? 1 : 2}
                >
                    <CoachAttentions
                        coach={coach}
                        segment={segment}
                        showNFCoach={showNFCoach}
                        withoutCoachNumberTitle={withoutCoachNumberTitle}
                        showCoachArrival={isDifferentCoachesArrival}
                        {...prepareQaAttributes(props)}
                    />
                    <CoachAutoSeatButton
                        className={cx('autoSeatButton')}
                        coach={coach}
                        passengers={passengers}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'autoSeatButton',
                        })}
                    />
                </Flex>
                <CoachFooter
                    className={cx('footer')}
                    coach={coach}
                    scale={scale}
                />
                <CoachNonRefundableDiscount
                    className={cx('nonRefundableDiscount')}
                    coach={coach}
                />
                <CoachTransportSchema
                    active={active}
                    coachSchemaRenderingDelay={coachSchemaRenderingDelay}
                    coach={coach}
                    scale={scale}
                    schema={schema}
                    gender={gender}
                    orderStep={orderStep}
                    passengers={passengers}
                    orderPlaces={orderPlaces}
                    onCoachChange={handleCoachChange}
                    schemeHasGroupedPlaces={schemeHasGroupedPlaces}
                    schemaPlaceMaskPrefix={schemaPlaceMaskPrefix}
                    onSchemaCatchError={handleSchemeCatchError}
                    {...prepareQaAttributes(props)}
                />
                {!isLast && !summaryIsVisible && (
                    <Separator
                        className={cx('separator', {
                            separator_topGap:
                                coachWithoutSchema || coach.directionConfirmed,
                        })}
                    />
                )}
                {summaryIsVisible && (
                    <>
                        <OrderSummaryCompact
                            above={5}
                            below={isLast ? undefined : 5}
                            buttonRef={passengersStepButtonRef}
                            {...prepareQaAttributes('orderSummaryCompact')}
                        />
                        {!isLast && <Separator margin={1} />}
                    </>
                )}
            </div>
        </li>
    );
};

export default React.memo(
    Coach,
    (prevProps: ICoachProps, nextProps: ICoachProps) =>
        !prevProps.active &&
        !nextProps.active &&
        prevProps.schema === nextProps.schema &&
        getCoachClassKey({coach: prevProps.coach}) ===
            getCoachClassKey({coach: nextProps.coach}),
);
